package com.qianhe.system.utils;

import com.qianhe.common.utils.StringUtils;
import org.gavaghan.geodesy.Ellipsoid;
import org.gavaghan.geodesy.GeodeticCalculator;
import org.gavaghan.geodesy.GeodeticCurve;
import org.gavaghan.geodesy.GlobalCoordinates;
import org.springframework.stereotype.Component;

/**
 * 坐标位置相关util
 */
@Component
public class PositionUtil {


    /**
     * 赤道半径
     */
    private static final double EQUATOR_RADIUS = 6378137;

    /**
     * 方法一：（反余弦计算方式）
     *
     * @param longitude1 第一个点的经度
     * @param latitude1  第一个点的纬度
     * @param longitude2 第二个点的经度
     * @param latitude2  第二个点的纬度
     * @return 返回距离，单位m
     */
    public static double getDistance1(double longitude1, double latitude1, double longitude2, double latitude2) {
        // 纬度
        double lat1 = Math.toRadians(latitude1);
        double lat2 = Math.toRadians(latitude2);
        // 经度
        double lon1 = Math.toRadians(longitude1);
        double lon2 = Math.toRadians(longitude2);
        // 纬度之差
        double a = lat1 - lat2;
        // 经度之差
        double b = lon1 - lon2;
        // 计算两点距离的公式
        double s = 2 * Math.asin(Math.sqrt(Math.pow(Math.sin(a / 2), 2) + Math.cos(lat1) * Math.cos(lat2) * Math.pow(Math.sin(b / 2), 2)));
        // 弧长乘赤道半径, 返回单位: 米
        s = s * EQUATOR_RADIUS;
        return s;
    }

    /**
     * 方法四：（利用第三方jar包计算）
     * 计算两个经纬度之间的距离
     *
     * @param longitude1 第一点的经度
     * @param latitude1  第一点的纬度
     * @param longitude2 第二点的经度
     * @param latitude2  第二点的纬度
     * @param ellipsoid  计算方式
     * @return 返回的距离，单位m
     */
    public static double getDistance(double longitude1, double latitude1, double longitude2, double latitude2, Ellipsoid ellipsoid) {
        // 创建GeodeticCalculator，调用计算方法，传入坐标系、经纬度用于计算距离
        GlobalCoordinates firstPoint = new GlobalCoordinates(latitude1, longitude1);
        GlobalCoordinates secondPoint = new GlobalCoordinates(latitude2, longitude2);
        GeodeticCurve geoCurve = new GeodeticCalculator().calculateGeodeticCurve(ellipsoid, firstPoint, secondPoint);
        return geoCurve.getEllipsoidalDistance();
    }

    /**
     * 判断坐标点是否在圆形区域内
     * 计算这个坐标点和圆心点之间的距离，然后跟圆的半径进行比较，如果比半径大，就不在圆形区域内，如果小于等于圆的半径，则该坐标点在圆形区域内
     *
     * @param longitude1 第一点的经度
     * @param latitude1  第一点的纬度
     * @param longitude2 第二点的经度
     * @param latitude2  第二点的纬度
     * @param radius     圆形范围半径（单位：米）
     * @return true：不在区域内; false:在区域内
     */
    public static boolean isInCircle(double longitude1, double latitude1, double longitude2, double latitude2, String radius) {
        if (StringUtils.isBlank(radius)) {
            throw new RuntimeException("请输入范围半径");
        }
        return getDistance(longitude1, latitude1, longitude2, latitude2,Ellipsoid.WGS84) > Double.parseDouble(radius);
    }
}
