package com.qianhe.system.service.impl;

import com.qianhe.common.utils.DateUtils;
import com.qianhe.common.utils.SecurityUtils;
import com.qianhe.common.utils.StringUtils;
import com.qianhe.system.domain.MapPoint;
import com.qianhe.system.domain.WaterStation;
import com.qianhe.system.domain.WaterStationRegion;
import com.qianhe.system.domain.WaterStationUser;
import com.qianhe.system.mapper.WaterStationMapper;
import com.qianhe.system.mapper.WaterStationRegionMapper;
import com.qianhe.system.mapper.WaterStationUserMapper;
import com.qianhe.system.service.IWaterStationService;
import com.qianhe.system.utils.AMapUtils;
import com.qianhe.system.utils.GraphUtils;
import com.qianhe.system.utils.PositionUtil;
import com.qianhe.system.vo.WaterStationUserVo;
import com.qianhe.system.vo.WaterStationVo;
import org.gavaghan.geodesy.Ellipsoid;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 站点Service业务层处理
 *
 * @author qianhe
 * @date 2023-11-23
 */
@Service
public class WaterStationServiceImpl implements IWaterStationService
{
    @Autowired
    private WaterStationMapper waterStationMapper;
    @Autowired
    private WaterStationUserMapper waterStationUserMapper;
    @Autowired
    private WaterStationUserServiceImpl waterStationUserService;
    @Autowired
    private WaterStationRegionMapper waterStationRegionMapper;

    /**
     * 查询站点
     *
     * @param id 站点主键
     * @return 站点
     */
    @Override
    public WaterStationVo selectWaterStationById(Long id)
    {
        WaterStationVo waterStationVo = new WaterStationVo();
        WaterStation waterStation = waterStationMapper.selectWaterStationById(id);
        BeanUtils.copyProperties(waterStation,waterStationVo);
        //查询站点用户
        WaterStationUser waterStationUser = new WaterStationUser();
        waterStationUser.setStationId(id);
        List<WaterStationUserVo> waterStationUsers = waterStationUserMapper.selectWaterStationUserList(waterStationUser);
        if (waterStationUsers.size() > 0){
            waterStationVo.setWaterStationUserList(waterStationUsers);
        }
        //查询区域
        List<WaterStationRegion> regions=waterStationRegionMapper.selectWaterStationRegionByStationId(id);
        waterStationVo.setWaterStationRegionList(regions);
        return waterStationVo;
    }

    /**
     * 查询站点列表
     *
     * @param waterStation 站点
     * @return 站点
     */
    @Override
    public List<WaterStation> selectWaterStationList(WaterStation waterStation)
    {
        return waterStationMapper.selectWaterStationList(waterStation);
    }

    /**
     * 新增站点
     *
     * @param waterStation 站点
     * @return 结果
     */
    @Override
    @Transactional
    public int insertWaterStation(WaterStation waterStation)
    {
        WaterStation w = gaodeToTencent(waterStation.getStationLon(),waterStation.getStationLat());
        waterStation.setStationLonTen(w.getStationLonTen());
        waterStation.setStationLatTen(w.getStationLatTen());
        waterStation.setCreateUser(SecurityUtils.getLoginUser().getUsername());
        waterStation.setCreateTime(DateUtils.getNowDate());
        int i = waterStationMapper.insertWaterStation(waterStation);
        //TODO 保存站点区域
        if(waterStation.getWaterStationRegionList().size()>0){
            List<WaterStationRegion> waterStationRegionList = waterStation.getWaterStationRegionList();
            for(int k=0;k<waterStationRegionList.size();k++){
                waterStationRegionList.get(k).setPx(k);
                waterStationRegionList.get(k).setStationId(waterStation.getId());
            }
            waterStationRegionMapper.batchWaterStationRegion(waterStationRegionList);
        }
        return i;
    }

    /**
     * 将高德地图的经纬度坐标转换为腾讯地图的经纬度坐标
     *
     * @param lng 高德地图经度
     * @param lat 高德地图纬度
     * @return 腾讯地图经纬度数组，第一个元素为腾讯地图经度，第二个元素为腾讯地图纬度
     */
    public static WaterStation gaodeToTencent(double lng, double lat) {
        WaterStation w = new WaterStation();
        double tx_lng = lng + 0.0065, tx_lat = lat + 0.006;
        double x = tx_lng, y = tx_lat;
        double z = Math.sqrt(x * x + y * y) + 0.00002 * Math.sin(y * Math.PI);
        double theta = Math.atan2(y, x) + 0.000003 * Math.cos(x * Math.PI);
        tx_lng = z * Math.cos(theta);
        tx_lat = z * Math.sin(theta);
        w.setStationLonTen(tx_lng);
        w.setStationLatTen(tx_lat);
        return w;
    }



    /**
     * 修改站点
     *
     * @param waterStation 站点
     * @return 结果
     */
    @Override
    @Transactional
    public int updateWaterStation(WaterStation waterStation)
    {
        //删除该站点用户
//        waterStationMapper.deleteWaterStationUserByStationId(waterStation.getId());
        WaterStation w = gaodeToTencent(waterStation.getStationLon(),waterStation.getStationLat());
        waterStation.setStationLonTen(w.getStationLonTen());
        waterStation.setStationLatTen(w.getStationLatTen());
        //TODO 删除站点区域
        waterStationRegionMapper.deleteWaterStationRegionByStationId(waterStation.getId());
        //TODO 保存站点区域
        if(waterStation.getWaterStationRegionList().size()>0){
            List<WaterStationRegion> waterStationRegionList = waterStation.getWaterStationRegionList();
            for(int k=0;k<waterStationRegionList.size();k++){
                waterStationRegionList.get(k).setPx(k);
                waterStationRegionList.get(k).setStationId(waterStation.getId());
            }
            waterStationRegionMapper.batchWaterStationRegion(waterStationRegionList);
        }
        return waterStationMapper.updateWaterStation(waterStation);
    }

    private void insertWaterStationUser(WaterStationVo waterStationVo) {
        //送水工集合
        List<WaterStationUserVo> waterStationUserList = waterStationVo.getWaterStationUserList();

        List<WaterStationUserVo> waterStationUsers = new ArrayList<>();
        //站点主键id
        Long id = waterStationVo.getId();
        if (StringUtils.isNotNull(waterStationUserList) && waterStationUserList.size() > 0){
            for (WaterStationUserVo waterStationUser : waterStationUserList) {
                waterStationUser.setStationId(id);
                waterStationUser.setCreateTime(DateUtils.getNowDate());
                waterStationUser.setCreateUser("管理员");
                waterStationUsers.add(waterStationUser);
            }
        }
        if (waterStationUsers.size() > 0){
            //批量新增站点用户
            waterStationMapper.batchInsertStationUser(waterStationUsers);
        }
    }

    /**
     * 批量删除站点
     *
     * @param ids 需要删除的站点主键
     * @return 结果
     */
    @Override
    @Transactional
    public int deleteWaterStationByIds(Long[] ids)
    {
        //批量删除站点用户
        waterStationMapper.deleteWaterStationUserByStationIds(ids);
        return waterStationMapper.deleteWaterStationByIds(ids);
    }

    /**
     * 删除站点信息
     *
     * @param id 站点主键
     * @return 结果
     */
    @Override
    public int deleteWaterStationById(Long id)
    {
        return waterStationMapper.deleteWaterStationById(id);
    }

    /**
     * 修改是否营业
     * @param waterStation
     * @return
     */
    @Override
    public int updateIsOpen(WaterStation waterStation) {
        return waterStationMapper.updateWaterStation(waterStation);
    }

    /**
     * 查询站点下拉框
     * @return
     */
    @Override
    public List<WaterStationVo> getStationList(double lon, double lat, String stationName) {
        WaterStation waterStation1 = new WaterStation();
        if (StringUtils.isNotEmpty(stationName)){
            waterStation1.setStationName(stationName);
        }
        List<WaterStation> stationList = waterStationMapper.getStationList(waterStation1);
        List<WaterStationVo> waterStationVoList = new ArrayList<>();
        //计算距离
        for (WaterStation waterStation : stationList) {
            //计算距离
            double distance = PositionUtil.getDistance(lon, lat, waterStation.getStationLonTen(), waterStation.getStationLatTen(), Ellipsoid.WGS84);
            //换算成千米
            double distance1 = distance / 1000;
            WaterStationVo waterStationVo = new WaterStationVo();
            BeanUtils.copyProperties(waterStation,waterStationVo);
            waterStationVo.setDistance(distance1);
            waterStationVoList.add(waterStationVo);
        }
        //根据距离升序排序
        List<WaterStationVo> collect = waterStationVoList.stream().sorted(Comparator.comparing(WaterStationVo::getDistance)).collect(Collectors.toList());
        //根据营业状态降序排序
        List<WaterStationVo> waterStationVos = collect.stream().sorted(Comparator.comparing(WaterStationVo::getIsOpen).reversed()).collect(Collectors.toList());
        return waterStationVos;
    }

    /**
     * 查询最近站点
     * @param lon
     * @param lat
     * @return
     */
    @Override
    public WaterStationVo getRecentlyStation(double lon, double lat) {
        //查询所有站点信息
        List<WaterStation> waterStations = waterStationMapper.selectWaterStationList(new WaterStation());
        List<WaterStationVo> waterStationVoList = new ArrayList<>();
        //循环站点，计算符合距离的站点
        for (WaterStation waterStation : waterStations) {
            //判断用户和站点之前的距离是否在站点管辖范围之内
            boolean inCircle = PositionUtil.isInCircle(lon, lat, waterStation.getStationLonTen(), waterStation.getStationLatTen(), "5000");
            //计算距离
            double distance = PositionUtil.getDistance(lon, lat, waterStation.getStationLonTen(), waterStation.getStationLatTen(), Ellipsoid.WGS84);
            if (inCircle){
                WaterStationVo waterStationVo = new WaterStationVo();
                BeanUtils.copyProperties(waterStation,waterStationVo);
                waterStationVo.setDistance(distance);
                waterStationVoList.add(waterStationVo);
            }
        }
        //根据距离排序，选出最近的站点
        List<WaterStationVo> collect = waterStationVoList.stream().sorted(Comparator.comparing(WaterStationVo::getDistance)).collect(Collectors.toList());
        return collect.get(0);
    }

    @Override
    public WaterStationVo getRecentlyStation1(String address,String address1) {

        double[] doubles;

        if(address1.contains(address)){
            doubles = AMapUtils.addressToGPS(address1);
        }else {
            doubles = AMapUtils.addressToGPS(address+address1);
        }

        double lon = doubles[0];
        double lat = doubles[1];

        System.out.println(lon+"-------------"+lat);

        //查询所有站点信息
        WaterStation waterStation1 = new WaterStation();
        waterStation1.setIsOpen("1");
        List<WaterStation> waterStations = waterStationMapper.selectWaterStationList(waterStation1);
        List<WaterStationVo> waterStationVoList = new ArrayList<>();
        //循环站点，计算符合距离的站点
        for (WaterStation waterStation : waterStations) {

            //计算距离
            double distance = PositionUtil.getDistance1(lon, lat, waterStation.getStationLonTen(), waterStation.getStationLatTen());
            System.out.println("dis---------------"+distance);
            if(distance<5000){
                WaterStationVo waterStationVo = new WaterStationVo();
                BeanUtils.copyProperties(waterStation,waterStationVo);
                waterStationVo.setDistance(distance);
                waterStationVoList.add(waterStationVo);
            }
        }
        //根据距离排序，选出最近的站点
        List<WaterStationVo> collect = waterStationVoList.stream().sorted(Comparator.comparing(WaterStationVo::getDistance)).collect(Collectors.toList());

        WaterStationVo stationVo = new WaterStationVo();
        if(collect.size()>0){
            stationVo = collect.get(0);
        }

        return stationVo;
    }

    /**
     * 查询站点下拉框（PC端）
     * @return
     */
    @Override
    public List<WaterStation> getWaterStationList() {
        return waterStationMapper.getStationList(new WaterStation());
    }

    @Override
    public WaterStationVo getRecentlyStation2(String address, String address1) {
        double[] doubles;

        if(address1.contains(address)){
            doubles = AMapUtils.addressToGPS(address1);
        }else {
            doubles = AMapUtils.addressToGPS(address+address1);
        }

        double lon = doubles[0];
        double lat = doubles[1];
        MapPoint n1 = new MapPoint(lon,lat);
        //查询所有站点信息
        WaterStation waterStation1 = new WaterStation();
        waterStation1.setIsOpen("1");
        System.out.println(lon+"-------------"+lat);
        List<WaterStation> waterStations = waterStationMapper.selectWaterStationList(waterStation1);
        List<WaterStationRegion> waterStationRegionList = waterStationRegionMapper.selectWaterStationRegionList(new WaterStationRegion());
        WaterStationVo vo=new WaterStationVo();
        //循环站点，计算符合距离的站点
        for (WaterStation waterStation : waterStations) {
            //筛选当前站点的区域
            List<WaterStationRegion> collect = waterStationRegionList.stream().filter(item -> item.getStationId().toString().equals(waterStation.getId().toString())).sorted(Comparator.comparing(WaterStationRegion::getPx)).collect(Collectors.toList());
            if(collect.size()>0){
                List<MapPoint> list =new ArrayList<>();
                collect.forEach(it->{
                    MapPoint point=new MapPoint();
                    point.setLat(it.getLat().doubleValue());
                    point.setLng(it.getLon().doubleValue());
                    list.add(point);
                });
                MapPoint[]  array2 = list.toArray(new MapPoint[list.size()]);
                boolean pointInPolygon = GraphUtils.isPointInPolygon(n1, array2);
                if(pointInPolygon){
                    BeanUtils.copyProperties(waterStation,vo);
                    break;
                }
            }

        }
        return vo;
    }
}
