package com.ruoyi.common.utils;

import java.lang.management.ManagementFactory;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;

import org.apache.commons.lang3.time.DateFormatUtils;

/**
 * 时间工具类
 *
 * @author ruoyi
 */
public class DateUtils extends org.apache.commons.lang3.time.DateUtils
{
    public static String YYYY = "yyyy";

    public static String YYYY_MM = "yyyy-MM";

    public static String YYYY_MM_DD = "yyyy-MM-dd";

    public static String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

    public static String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    private static String[] parsePatterns = {
            "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
            "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM",
            "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm", "yyyy.MM"};

    /**
     * 获取当前Date型日期
     *
     * @return Date() 当前日期
     */
    public static Date getNowDate()
    {
        return new Date();
    }

    /**
     * 获取当前日期, 默认格式为yyyy-MM-dd
     *
     * @return String
     */
    public static String getDate()
    {
        return dateTimeNow(YYYY_MM_DD);
    }

    public static final String getTime()
    {
        return dateTimeNow(YYYY_MM_DD_HH_MM_SS);
    }

    public static final String dateTimeNow()
    {
        return dateTimeNow(YYYYMMDDHHMMSS);
    }

    public static final String dateTimeNow(final String format)
    {
        return parseDateToStr(format, new Date());
    }

    public static final String dateTime(final Date date)
    {
        return parseDateToStr(YYYY_MM_DD, date);
    }

    public static final String parseDateToStr(final String format, final Date date)
    {
        return new SimpleDateFormat(format).format(date);
    }

    public static final Date dateTime(final String format, final String ts)
    {
        try
        {
            return new SimpleDateFormat(format).parse(ts);
        }
        catch (ParseException e)
        {
            throw new RuntimeException(e);
        }
    }

    /**
     * 日期路径 即年/月/日 如2018/08/08
     */
    public static final String datePath()
    {
        Date now = new Date();
        return DateFormatUtils.format(now, "yyyy/MM/dd");
    }

    /**
     * 日期路径 即年/月/日 如20180808
     */
    public static final String dateTime()
    {
        Date now = new Date();
        return DateFormatUtils.format(now, "yyyyMMdd");
    }

    /**
     * 日期型字符串转化为日期 格式
     */
    public static Date parseDate(Object str)
    {
        if (str == null)
        {
            return null;
        }
        try
        {
            return parseDate(str.toString(), parsePatterns);
        }
        catch (ParseException e)
        {
            return null;
        }
    }

    /**
     * 获取服务器启动时间
     */
    public static Date getServerStartDate()
    {
        long time = ManagementFactory.getRuntimeMXBean().getStartTime();
        return new Date(time);
    }

    /**
     * 计算相差天数
     */
    public static int differentDaysByMillisecond(Date date1, Date date2)
    {
        return Math.abs((int) ((date2.getTime() - date1.getTime()) / (1000 * 3600 * 24)));
    }

    /**
     * 计算两个时间差
     */
    public static String getDatePoor(Date endDate, Date nowDate)
    {
        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        // long ns = 1000;
        // 获得两个时间的毫秒时间差异
        long diff = endDate.getTime() - nowDate.getTime();
        // 计算差多少天
        long day = diff / nd;
        // 计算差多少小时
        long hour = diff % nd / nh;
        // 计算差多少分钟
        long min = diff % nd % nh / nm;
        // 计算差多少秒//输出结果
        // long sec = diff % nd % nh % nm / ns;
        return day + "天" + hour + "小时" + min + "分钟";
    }

    /**
     * 增加 LocalDateTime ==> Date
     */
    public static Date toDate(LocalDateTime temporalAccessor)
    {
        ZonedDateTime zdt = temporalAccessor.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    /**
     * 增加 LocalDate ==> Date
     */
    public static Date toDate(LocalDate temporalAccessor)
    {
        LocalDateTime localDateTime = LocalDateTime.of(temporalAccessor, LocalTime.of(0, 0, 0));
        ZonedDateTime zdt = localDateTime.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    /**
     * 获取星期几
     * */
    public static String getWeekday(){
        String[] weekDays = {"星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六"};
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        //因为数组下标从0开始，而返回的是数组的内容，是数组{1,2,3,4,5,6,7}中用1~7来表示，所以要减1
        int week = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (week < 0) {
            week = 0;
        }
        return weekDays[week];
    }


    /**
     * 根据日期获取 星期
     *
     * @param datetime
     * @return
     */
    public static String dateToWeek(String datetime) {

        SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd");
        String[] weekDays = {"星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六"};
        Calendar cal = Calendar.getInstance();
        Date date;
        try {
            date = f.parse(datetime);
            cal.setTime(date);
        } catch (ParseException e) {
//            e.printStackTrace();
            date = null;
        }
        //一周的第几天
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0)
            w = 0;
        return weekDays[w];
    }




    /**
     * 通过开始和结束日期拿到中间所有的日期（包括开始和结束）
     * @param startDate
     * @param endDate
     * @return
     */
    public static List<String> getDutyList(Date startDate, Date endDate){
        List<String> days=new ArrayList<String>();
        Calendar startTime=Calendar.getInstance();
        Calendar endTime=Calendar.getInstance();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        try {
            //将开始日期设置给calendar
            String start = sdf.format(startDate);
            startTime.setTime(sdf.parse(start));
//            System.out.println("开始日期="+sdf.format(startTime.getTime()));
            startTime.add(Calendar.DATE,-1);//日期减1，包含开始日期
            //将结束日期设置给calendar
            String end = sdf.format(endDate);
            endTime.setTime(sdf.parse((end)));
//            System.out.println("结束日期="+sdf.format(endTime.getTime()));
            while (startTime.before(endTime)){
                startTime.add(Calendar.DAY_OF_MONTH,1);
                days.add(sdf.format(startTime.getTime()));
            }
//            System.out.println("days="+days.toString());
        }catch (Exception e){
            e.printStackTrace();
        }
        return days;
    }

    /**
     * 传入2022-01-01
     * 传出2022年01月01日
     * **/
    public static String formatDate(Date date){
        String d =  String.format("%tY年%tm月%td日", date,date,date);
        return d;
    }

    /**
     * 判断当前时间是否在时间范围内
     *
     * @param nowTime
     * @param startTime
     * @param endTime
     * @return boolean
     * @author zxzhang
     * @date 2020/3/16
     */
    public static boolean isEffectiveDate(Date nowTime, Date startTime, Date endTime) {
        if (nowTime.getTime() == startTime.getTime()
                || nowTime.getTime() == endTime.getTime()) {
            return true;
        }

        Calendar date = Calendar.getInstance();
        date.setTime(nowTime);

        Calendar begin = Calendar.getInstance();
        begin.setTime(startTime);

        Calendar end = Calendar.getInstance();
        end.setTime(endTime);

        if (date.after(begin) && date.before(end)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 根据传入的日期获取所在学年
     * d yyyy-MM-dd
     * */
    public static int getSchoolYear(Date d) {
        /* 如果在8、9、10、11、12、1月，为此学年第 2 学期，
         *     其中在9、10、11、12月为 year 学年，1月为 year-1 学年。
         * 如果在2、3、4、5、6、7、月，为此学年第 2 学期，
         *     year-1学年。
         */
        //获取日期参数的年月  假设 2021-08-26
        Calendar ca = Calendar.getInstance();
        ca.setTime(d);
        int month = ca.get(Calendar.MONTH)+1;//第几个月 8
        int year = ca.get(Calendar.YEAR);//年份数值 2021
        int term = 1;
        if(month < 8){
            year = year - 1;
        }
        if (month>=2 && month<8){
            term = 2;
        } else{
            term = 1;
        }
        return year;
    }

    //根据传入的日期获取所在学期
    public static int getTerm(Date d) {
        /* 如果在8、9、10、11、12、1月，为此学年第 2 学期，
         *     其中在9、10、11、12月为 year 学年，1月为 year-1 学年。
         * 如果在2、3、4、5、6、7、月，为此学年第 2 学期，
         *     year-1学年。
         */
        //获取日期参数的年月  假设 2021-08-26
        Calendar ca = Calendar.getInstance();
        ca.setTime(d);
        int month = ca.get(Calendar.MONTH)+1;//第几个月 8
        int year = ca.get(Calendar.YEAR);//年份数值 2021
        int term = 1;
        if(month < 8){
            year = year - 1;
        }
        if (month>=2 && month<8){
            term = 2;
        } else{
            term = 1;
        }
        return term;
    }

    public static TermRange getTermRange(String year, String noticeSemester){
        TermRange termRange = new TermRange();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        switch (noticeSemester){
            case "上学期":
            case "上":
                Calendar startDay = Calendar.getInstance();
                startDay.set(Integer.parseInt(year), 7, 1);
                termRange.setStartDay(sdf.format(startDay.getTime()));
                Calendar endDay = Calendar.getInstance();
                endDay.set(Integer.parseInt(year)+1, 0, 31);
                termRange.setEndDay(sdf.format(endDay.getTime()));
                break;
            case "下学期":
            case "下":
                Calendar xstartDay = Calendar.getInstance();
                xstartDay.set(Integer.parseInt(year), 1, 1);
                termRange.setStartDay(sdf.format(xstartDay.getTime()));
                Calendar xendDay = Calendar.getInstance();
                xendDay.set(Integer.parseInt(year), 6, 31);
                termRange.setEndDay(sdf.format(xendDay.getTime()));
                break;
        }
        return termRange;
    }

    public static void main(String[] args) {
        System.out.println(getSchoolYear(getNowDate()));;
        System.out.println(getTerm(getNowDate()));;

    }
}
