package yangtz.cs.liu.campus.service.impl.recruit;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.core.utils.StringUtil;
import com.ruoyi.common.exception.ServiceException;
import com.ruoyi.common.utils.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import yangtz.cs.liu.campus.constant.RecruitExamConstant;
import yangtz.cs.liu.campus.domain.recruit.SchoolRecruitExam;
import yangtz.cs.liu.campus.domain.recruit.SchoolRecruitRelation;
import yangtz.cs.liu.campus.mapper.recruit.SchoolRecruitExamMapper;
import yangtz.cs.liu.campus.mapper.recruit.SchoolRecruitRelationMapper;
import yangtz.cs.liu.campus.service.recruit.ISchoolRecruitRelationService;
import yangtz.cs.liu.campus.vo.recruit.RecruitExamAddressVo;
import yangtz.cs.liu.campus.vo.recruit.RecruitExamScoreVo;

import java.util.List;

@Service
public class SchoolRecruitRelationServiceImpl extends ServiceImpl<SchoolRecruitRelationMapper, SchoolRecruitRelation> implements ISchoolRecruitRelationService {
    @Autowired
    private SchoolRecruitRelationMapper recruitRelationMapper;

    @Autowired
    private SchoolRecruitExamMapper recruitExamMapper;

    @Override
    public SchoolRecruitRelation getRelationByIds(Long examId, Long studentId) {
        return recruitRelationMapper.getRelationByIds(examId, studentId);
    }

    @Override
    public List<RecruitExamAddressVo> getAddressList(RecruitExamAddressVo vo) {
        return recruitRelationMapper.getAddressList(vo);
    }

    @Override
    public RecruitExamScoreVo getAddressInfo(Long id) {
        return recruitRelationMapper.getAddressInfo(id);
    }

    /**
     * 考试管理-考试管理-列表
     * */
    @Override
    public List<RecruitExamScoreVo> getScoreList(RecruitExamScoreVo vo) {
        return recruitRelationMapper.getScoreList(vo);
    }

    //校验导入的数据
    @Override
    public void checkImportAddress(List<RecruitExamAddressVo> list) {
        //检验是否为空
        if (StringUtils.isNull(list) || list.size() == 0) {
            throw new ServiceException("导入数据不能为空");
        }

        // 准备记录日志数据
        int failureNum = 0;
        StringBuilder failureMsg = new StringBuilder();
        for (RecruitExamAddressVo vo : list) {
            if (StringUtils.isEmpty(vo.getIdCard())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、身份证号为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getExamNumber())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、准考证号为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getInGroup())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、分组为空,请您重新输入! ");
            }
            if (StringUtils.isNull(vo.getExamDate())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、考试日期为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getExamTime())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、考试时间为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getExamAddress())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、考试地点为空,请您重新输入! ");
            }
        }
        if (failureNum > 0) {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new ServiceException(failureMsg.toString());
        }
    }

    /**
     * 导入考场数据
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String importDataAddress(List<RecruitExamAddressVo> list, Long examId) {
        // 准备记录日志数据
        int successNum = 0;
        StringBuilder successMsg = new StringBuilder();
        for (RecruitExamAddressVo vo : list) {
            if (vo.getExamTime().contains("：")) {
                String t = vo.getExamTime();
                t = t.replace("：", ":");
                vo.setExamTime(t);
            }
            LambdaQueryWrapper<SchoolRecruitRelation> lqw = new LambdaQueryWrapper<>();
            lqw.eq(SchoolRecruitRelation::getInfoIdCard, StringUtil.clearSpace(vo.getIdCard())).eq(SchoolRecruitRelation::getExamId, examId);
            List<SchoolRecruitRelation> studentList = recruitRelationMapper.selectList(lqw);
            if(StringUtils.isNull(studentList) || studentList.size() == 0){
                throw new ServiceException("查不到身份证号为" + vo.getIdCard() + "的学生信息，请检查后重新导入");
            }
            SchoolRecruitRelation student = studentList.get(0);
            student.setExamNumber(vo.getExamNumber());
            student.setInGroup(vo.getInGroup());
            student.setExamDate(vo.getExamDate());
            student.setExamTime(vo.getExamTime());
            student.setExamAddress(vo.getExamAddress());
            student.update();
            recruitRelationMapper.updateById(student);
            successNum++;
        }
        successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条");
        return successMsg.toString();
    }

    /**
     * 考试管理-考场分配-结束
     * 有考场分配信息为空的要进行提示
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int addressEnd(Long examId) {
        RecruitExamAddressVo vo = new RecruitExamAddressVo();
        vo.setExamId(examId);
        vo.setAuditStatus(RecruitExamConstant.PASS);
        List<RecruitExamAddressVo> list =  recruitRelationMapper.getAddressList(vo);
        if(StringUtils.isNull(list) && list.size() == 0){
            throw new ServiceException("考生信息为空，不允许结束");
        }else {
            for(RecruitExamAddressVo addressVo : list){
                if(StringUtils.isEmpty(addressVo.getExamNumber())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的准考证号为空，请检查");
                }
                if(StringUtils.isEmpty(addressVo.getInGroup())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的分组为空，请检查");
                }
                if(StringUtils.isNull(addressVo.getExamDate())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的考试日期为空，请检查");
                }
                if(StringUtils.isEmpty(addressVo.getExamTime())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的考试时间为空，请检查");
                }
                if(StringUtils.isEmpty(addressVo.getExamAddress())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的考试地点为空，请检查");
                }
            }
            SchoolRecruitExam exam = recruitExamMapper.selectById(examId);
            exam.setStatus(RecruitExamConstant.SCORE);
            return recruitExamMapper.updateById(exam);
        }
    }

    //校验导入的成绩信息
    @Override
    public void checkImportScore(List<RecruitExamScoreVo> list) {
        //检验是否为空
        if (StringUtils.isNull(list) || list.size() == 0) {
            throw new ServiceException("导入数据不能为空");
        }

        // 准备记录日志数据
        int failureNum = 0;
        StringBuilder failureMsg = new StringBuilder();
        for (RecruitExamScoreVo vo : list) {
            if (StringUtils.isEmpty(vo.getIdCard())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、身份证号为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getScore())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、成绩为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getResult())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、结果为空,请您重新输入! ");
            }
            if (StringUtils.isEmpty(vo.getPassScore())) {
                failureNum++;
                failureMsg.append("<br/>" + failureNum + "、资格线为空,请您重新输入! ");
            }
        }
        if (failureNum > 0) {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new ServiceException(failureMsg.toString());
        }
    }

    /**
     * 导入成绩数据
     * */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String importDataScore(List<RecruitExamScoreVo> list, Long examId) {
        // 准备记录日志数据
        int successNum = 0;
        StringBuilder successMsg = new StringBuilder();
        for (RecruitExamScoreVo vo : list) {
            LambdaQueryWrapper<SchoolRecruitRelation> lqw = new LambdaQueryWrapper<>();
            lqw.eq(SchoolRecruitRelation::getInfoIdCard, vo.getIdCard()).eq(SchoolRecruitRelation::getExamId, examId);
            List<SchoolRecruitRelation> studentList = recruitRelationMapper.selectList(lqw);
            if(StringUtils.isNull(studentList) || studentList.size() == 0){
                throw new ServiceException("查不到身份证号为" + vo.getIdCard() + "的学生信息，请检查后重新导入");
            }
            SchoolRecruitRelation student = studentList.get(0);
            student.setScore(vo.getScore());
            student.setResult(vo.getResult());
            student.setPassScore(vo.getPassScore());
            student.update();
            recruitRelationMapper.updateById(student);
            successNum++;
        }
        successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条");
        return successMsg.toString();
    }

    /**
     * 考试管理-成绩发布-结束
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int scoreEnd(Long examId) {
        RecruitExamScoreVo vo = new RecruitExamScoreVo();
        vo.setExamId(examId);
        vo.setAuditStatus(RecruitExamConstant.PASS);
        List<RecruitExamScoreVo> list =  recruitRelationMapper.getScoreList(vo);
        if(StringUtils.isNull(list) && list.size() == 0){
            throw new ServiceException("考生信息为空，不允许结束");
        }else {
            for(RecruitExamScoreVo addressVo : list){
                if(StringUtils.isEmpty(addressVo.getScore())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的成绩为空，请检查");
                }
                if(StringUtils.isEmpty(addressVo.getResult())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的结果为空，请检查");
                }
                if(StringUtils.isNull(addressVo.getPassScore())){
                    throw new ServiceException("身份证号为" + addressVo.getIdCard() + "的考生的资格线为空，请检查");
                }
            }
            SchoolRecruitExam exam = recruitExamMapper.selectById(examId);
            exam.setStatus(RecruitExamConstant.END);
            return recruitExamMapper.updateById(exam);
        }
    }
}
