package yangtz.cs.liu.campus.aspectj;

import com.core.annotation.DeptFilter;
import com.core.domain.OurBaseEntity;
import com.ruoyi.common.core.domain.entity.SysDept;
import com.ruoyi.common.core.domain.entity.SysRole;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.domain.model.LoginUser;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.common.utils.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.stereotype.Component;

@Aspect
@Component
public class DeptFilterAspectj {

    /**
     * 全部数据权限
     */
    public static final String DATA_SCOPE_ALL = "1";

    /**
     * 自定数据权限
     */
    public static final String DATA_SCOPE_CUSTOM = "2";

    /**
     * 部门数据权限
     */
    public static final String DATA_SCOPE_DEPT = "3";

    /**
     * 部门及以下数据权限
     */
    public static final String DATA_SCOPE_DEPT_AND_CHILD = "4";

    /**
     * 仅本人数据权限
     */
    public static final String DATA_SCOPE_SELF = "5";

    /**
     * 部门上一级以下数据权限
     */
    public static final String DATA_SCOPE_DEPT_AND_UPPER = "8";

    public static final String DATA_SCOPE = "dataScope";

    @Before("@annotation(controllerDeptFilter)")
    public void doBefore(JoinPoint point, DeptFilter controllerDeptFilter) throws Throwable
    {
        clearDataScope(point);
        handleDataScope(point, controllerDeptFilter);
    }

    protected void handleDataScope(final JoinPoint joinPoint, DeptFilter controllerDataScope)
    {
        // 获取当前的用户
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (StringUtils.isNotNull(loginUser))
        {
            SysUser currentUser = loginUser.getUser();
            // 如果是超级管理员，则不过滤数据
            if (StringUtils.isNotNull(currentUser) && !currentUser.isAdmin())
            {
                dataScopeFilter(joinPoint, currentUser, controllerDataScope.deptAlias(),
                        controllerDataScope.userAlias());
            }
        }
    }

    /**
     * 数据范围过滤
     *
     * @param joinPoint 切点
     * @param user 用户
     * @param userAlias 别名
     */
    public static void dataScopeFilter(JoinPoint joinPoint, SysUser user, String deptAlias, String userAlias)
    {
        StringBuilder sqlString = new StringBuilder();
        //该用户可能有多个角色，也就是说多个身份
        for (SysRole role : user.getRoles())
        {
            String dataScope = role.getDataScope();//拿数据范围
            if (DATA_SCOPE_ALL.equals(dataScope))
            {
                sqlString = new StringBuilder();
                break;
            }
            else if (DATA_SCOPE_CUSTOM.equals(dataScope))
            {
                sqlString.append(StringUtils.format(
                        " OR {}.dept_id IN ( SELECT dept_id FROM sys_role_dept WHERE role_id = {} ) ", deptAlias,
                        role.getRoleId()));
            }
            else if (DATA_SCOPE_DEPT.equals(dataScope))
            {
                sqlString.append(StringUtils.format(" OR {}.dept_id = {} ", deptAlias, user.getDeptId()));
            }
            else if (DATA_SCOPE_DEPT_AND_CHILD.equals(dataScope)) //部门及以下
            {
                sqlString.append(StringUtils.format(
                        " OR {}.dept_id IN ( SELECT dept_id FROM sys_dept WHERE dept_id = {} or find_in_set( {} , ancestors ) )",
                        deptAlias, user.getDeptId(), user.getDeptId()));
            }
            else if (DATA_SCOPE_SELF.equals(dataScope))
            {
                if (StringUtils.isNotBlank(userAlias))
                {
                    sqlString.append(StringUtils.format(" OR {}.user_id = {} ", userAlias, user.getUserId()));
                }
                else
                {
                    // 数据权限为仅本人且没有userAlias别名不查询任何数据
                    sqlString.append(" OR 1=0 ");
                }
            }
            //当前用户数据权限等于8
            else if (DATA_SCOPE_DEPT_AND_UPPER.equals(dataScope)) {
                //拿到当前用户所在部门
                SysDept dept = user.getDept();
                //拿到父节点
                Long parentId = dept.getParentId();
                //拼接sql语句
                sqlString.append(StringUtils.format(
                        " OR {}.dept_id IN ( SELECT dept_id FROM sys_dept WHERE dept_id = {} or find_in_set( {} , ancestors ) )",
                        deptAlias, parentId, parentId));
            }
        }

        if (StringUtils.isNotBlank(sqlString.toString()))
        {
            Object params = joinPoint.getArgs()[0];
            if (StringUtils.isNotNull(params) && params instanceof OurBaseEntity)
            {
                OurBaseEntity baseEntity = (OurBaseEntity) params;
                baseEntity.getParams().put(DATA_SCOPE, " AND (" + sqlString.substring(4) + ")");
            }
            else if (StringUtils.isNotNull(params) && params instanceof OurBaseEntity)
            {
                OurBaseEntity basicEntity = (OurBaseEntity) params;
                basicEntity.getParams().put(DATA_SCOPE, " AND (" + sqlString.substring(4) + ")");
            }
        }
    }

    /**
     * 拼接权限sql前先清空params.dataScope参数防止注入
     */
    private void clearDataScope(final JoinPoint joinPoint)
    {
        Object params = joinPoint.getArgs()[0];
        if (StringUtils.isNotNull(params) && params instanceof OurBaseEntity)
        {
            OurBaseEntity baseEntity = (OurBaseEntity) params;
            baseEntity.getParams().put(DATA_SCOPE, "");
        }
        else if (StringUtils.isNotNull(params) && params instanceof OurBaseEntity)
        {
            OurBaseEntity basicEntity = (OurBaseEntity) params;
            basicEntity.getParams().put(DATA_SCOPE, "");
        }
    }

    //    protected void handleDataScope(final JoinPoint joinPoint, DeptFilter controllerDeptFilter) {
//        StringBuilder sqlString = new StringBuilder();
//        SysUser user = SecurityUtils.getLoginUser().getUser();
//        Long deptId = user.getDeptId();
//        if ("d".equals(controllerDeptFilter.deptAlias())) {
//            sqlString.append(StringUtils.format(
//                    "{}.dept_id IN ( SELECT dept_id FROM sys_dept WHERE dept_id = {} or find_in_set( {} , ancestors ) )",
//                    "d", deptId, deptId));
//        } else if ("d".equals(controllerDeptFilter.deptOnlyAlias())) {
//            sqlString.append(StringUtils.format(
//                    "{}.dept_id = {}",
//                    "d", deptId));
//        } else if ("d".equals(controllerDeptFilter.deptUpperAlias())) {
//            //拿到当前用户所在部门
//            SysDept dept = user.getDept();
//            //拿到父节点
//            Long parentId = dept.getParentId();
//            sqlString.append(StringUtils.format(
//                    "{}.dept_id IN ( SELECT dept_id FROM sys_dept WHERE dept_id = {} or find_in_set( {} , ancestors ) )",
//                    "d", parentId, parentId));
//        }
//        if (StringUtils.isNotBlank(sqlString.toString()))
//        {
//            Object params = joinPoint.getArgs()[0];
//            if (StringUtils.isNotNull(params) && params instanceof OurBaseEntity )
//            {
//                OurBaseEntity baseEntity = (OurBaseEntity) params;
//                baseEntity.getParams().put(DATA_SCOPE, " AND (" + sqlString + ")");
//            }
//            else if (StringUtils.isNotNull(params) && params instanceof BasicEntity)
//            {
//                BasicEntity basicEntity = (BasicEntity) params;
//                basicEntity.getParams().put(DATA_SCOPE, " AND (" + sqlString + ")");
//            }
//        }
//    }
}
