package com.ruoyi.framework.web.service;

import com.ruoyi.common.constant.Constants;
import com.ruoyi.common.core.domain.model.LoginUser;
import com.ruoyi.common.core.domain.model.MpLoginUser;
import com.ruoyi.common.core.domain.model.WebLoginBody;
import com.ruoyi.common.core.redis.RedisCache;
import com.ruoyi.common.utils.ServletUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.ip.AddressUtils;
import com.ruoyi.common.utils.ip.IpUtils;
import com.ruoyi.common.utils.uuid.IdUtils;
import eu.bitwalker.useragentutils.UserAgent;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

@Component
public class WebTokenService {

    // 令牌自定义标识
    @Value("${token.header}")
    private String header;

    // 令牌有效期（默认30分钟）
    @Value("${token.expireTime}")
    private int expireTime;

    // 令牌秘钥
    @Value("${token.secret}")
    private String secret;

    protected static final long MILLIS_SECOND = 1000;

    protected static final long MILLIS_MINUTE = 60 * MILLIS_SECOND;

    private static final Long MILLIS_MINUTE_TEN = 20 * 60 * 1000L;

    private static final String WEB_PREFIX = "Bearer ";

    @Autowired
    private RedisCache redisCache;

    /**
     * 获取请求token
     *
     * @param request
     * @return token
     */
    public String getToken(HttpServletRequest request)
    {
        String token = request.getHeader(header);
        if (StringUtils.isNotEmpty(token) && token.startsWith(WEB_PREFIX))
        {
            return token.replace(WEB_PREFIX, "");
        }
        return null;
    }

    /**
     * 创建令牌
     *
     * @param loginBody 艺考网页考生信息
     * @return 令牌
     */
    public String createToken(WebLoginBody loginBody)
    {
        String webToken = IdUtils.fastUUID();
        loginBody.setWebToken(webToken);
        setUserAgent(loginBody);
        refreshToken(loginBody);

        Map<String, Object> claims = new HashMap<>();
        claims.put(WEB_PREFIX, webToken);
        return createToken(claims);
    }

    /**
     * 从数据声明生成令牌
     *
     * @param claims 数据声明
     * @return 令牌
     */
    private String createToken(Map<String, Object> claims)
    {
        String token = Jwts.builder()
                .setClaims(claims)
                .signWith(SignatureAlgorithm.HS512, secret).compact();
        return token;
    }

    /**
     * 设置用户代理信息
     *
     * @param loginBody 登录信息
     */
    public void setUserAgent(WebLoginBody loginBody)
    {
        UserAgent userAgent = UserAgent.parseUserAgentString(ServletUtils.getRequest().getHeader("User-Agent"));
        String ip = IpUtils.getIpAddr(ServletUtils.getRequest());
        loginBody.setIpaddr(ip);
        loginBody.setLoginLocation(AddressUtils.getRealAddressByIP(ip));
        loginBody.setBrowser(userAgent.getBrowser().getName());
        loginBody.setOs(userAgent.getOperatingSystem().getName());
    }

    /**
     * 刷新令牌有效期
     *
     * @param loginBody 登录信息
     */
    public void refreshToken(WebLoginBody loginBody)
    {
        loginBody.setLoginTime(System.currentTimeMillis());
        loginBody.setExpireTime(loginBody.getLoginTime() + expireTime * MILLIS_MINUTE);
        // 根据uuid将loginUser缓存
        String userKey = getTokenKey(loginBody.getWebToken());
        redisCache.setCacheObject(userKey, loginBody, expireTime, TimeUnit.MINUTES);
    }

    private String getTokenKey(String uuid)
    {
        return "web_token:" + uuid;
    }

    /**
     * 获取用户身份信息
     * @return 用户信息
     */
    public WebLoginBody getWebLoginBody(String token)
    {
        // 获取请求携带的令牌
        if (StringUtils.isNotEmpty(token))
        {
            try
            {
                Claims claims = parseToken(token);
                // 解析对应的权限以及用户信息
                String uuid = (String) claims.get(WEB_PREFIX);
                String userKey = getTokenKey(uuid);
                WebLoginBody webLoginBody = redisCache.getCacheObject(userKey);
                return webLoginBody;
            }
            catch (Exception e)
            {
            }
        }
        return null;
    }

    /**
     * 从令牌中获取数据声明
     *
     * @param token 令牌
     * @return 数据声明
     */
    private Claims parseToken(String token)
    {
        return Jwts.parser()
                .setSigningKey(secret)
                .parseClaimsJws(token)
                .getBody();
    }

    /**
     * 验证令牌有效期，相差不足20分钟，自动刷新缓存
     *
     * @param webLoginBody
     * @return 令牌
     */
    public void verifyToken(WebLoginBody webLoginBody)
    {
        long expireTime = webLoginBody.getExpireTime();
        long currentTime = System.currentTimeMillis();
        if (expireTime - currentTime <= MILLIS_MINUTE_TEN)
        {
            refreshToken(webLoginBody);
        }
    }

    /**
     * 删除用户身份信息
     */
    public void delWebLoginBody(String token)
    {
        if (StringUtils.isNotEmpty(token))
        {
            String userKey = getTokenKey(token);
            redisCache.deleteObject(userKey);
        }
    }
}
