package com.zjsgfa.project.zjsgfa.util;

import org.apache.commons.codec.binary.Base64;
import org.apache.batik.transcoder.TranscoderException;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.image.PNGTranscoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;

public class ImageProcessUtil {
    private static final Logger logger = LoggerFactory.getLogger(ImageProcessUtil.class);

    /**
     * 从完整Base64字符串中提取并解码SVG内容
     */
    public static String extractSvgFromBase64(String fullBase64) {
        if (fullBase64 == null || !fullBase64.startsWith("data:image/svg+xml")) {
            logger.error("无效的SVG Base64格式");
            return null;
        }

        try {
            // 移除前缀（兼容带charset的格式）
            String base64Content = fullBase64.replaceAll("^data:image/svg\\+xml;.*?base64,", "");
            
            // 验证Base64格式
            if (!Base64.isBase64(base64Content)) {
                logger.error("Base64字符串格式无效");
                return null;
            }
            
            // 解码为SVG字符串
            byte[] svgBytes = Base64.decodeBase64(base64Content);
            return new String(svgBytes, "UTF-8");
        } catch (Exception e) {
            logger.error("提取SVG内容失败", e);
            return null;
        }
    }

    /**
     * 将SVG字符串转换为PNG输入流（确保字节流有效）
     */
    public static InputStream convertSvgToPng(String svgContent) {
        if (svgContent == null || svgContent.trim().isEmpty()) {
            logger.error("SVG内容为空");
            return null;
        }

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            // 初始化Batik转换器
            PNGTranscoder transcoder = new PNGTranscoder();
            TranscoderInput input = new TranscoderInput(
                new ByteArrayInputStream(svgContent.getBytes("UTF-8"))
            );
            TranscoderOutput output = new TranscoderOutput(outputStream);

            // 执行转换
            transcoder.transcode(input, output);
            outputStream.flush();

            // 验证转换结果
            byte[] pngBytes = outputStream.toByteArray();
            if (pngBytes.length == 0) {
                logger.error("转换后的PNG字节流为空");
                return null;
            }

            return new ByteArrayInputStream(pngBytes);
        } catch (TranscoderException e) {
            logger.error("SVG转换PNG失败（可能包含不支持的元素）", e);
            return null;
        } catch (Exception e) {
            logger.error("PNG流处理失败", e);
            return null;
        }
    }


    /**
     * 将data:image/png;base64格式的字符串转换为PNG输入流
     * @param base64Str 包含data:image/png;base64前缀的base64字符串
     * @return 有效的PNG字节输入流
     * @throws IllegalArgumentException 输入字符串格式错误或解码失败时抛出
     */
    public static InputStream convertBase64ToPngInputStream(String base64Str) {
        // 1. 校验输入是否为空
        if (base64Str == null || base64Str.isEmpty()) {
            throw new IllegalArgumentException("Base64字符串不能为空");
        }

        // 2. 剥离data:image/png;base64前缀
        String base64Prefix = "data:image/png;base64,";
        if (!base64Str.startsWith(base64Prefix)) {
            throw new IllegalArgumentException("输入字符串不是有效的PNG base64格式（缺少前缀）");
        }
        String pureBase64Str = base64Str.substring(base64Prefix.length());

        try {
            // 3. Base64解码得到PNG原始字节数组
            byte[] pngBytes = Base64.decodeBase64(pureBase64Str);

            // 4. 包装为ByteArrayInputStream（输入流）
            return new ByteArrayInputStream(pngBytes);
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Base64解码失败，输入字符串可能无效", e);
        }
    }




}
    