package com.zjsgfa.project.zjsgfa.util;

import com.zjsgfa.common.utils.StringUtils;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 钻井系统菜单与井号识别工具
 * 功能：从用户自然语言中识别目标菜单和井号
 */
public class DrillingMenuRecognizer {

    // 定义菜单-关键词映射（静态常量，初始化一次即可）
    private static final Map<String, List<String>> MENU_KEYWORDS;

    // 静态代码块初始化菜单关键词映射
    static {
        MENU_KEYWORDS = new HashMap<>();
        // 逐个添加菜单和对应的关键词
        MENU_KEYWORDS.put("钻头单支分析", Arrays.asList("钻头单支", "单支钻头","单支", "分析", "单支钻头分析"));
        MENU_KEYWORDS.put("口井周期分析", Arrays.asList("周期", "井周期","周期分析","口井周期分析", "周期数据", "口井周期"));
        MENU_KEYWORDS.put("钻头区块分析", Arrays.asList("钻头区块", "区块钻头","区块", "区块钻头分析"));
        MENU_KEYWORDS.put("钻具组合分析", Arrays.asList("钻具组合", "组合钻具","钻具", "钻具组合分析"));
        MENU_KEYWORDS.put("邻井查询", Arrays.asList("邻井", "邻井查询", "周边井", "相邻井"));
        MENU_KEYWORDS.put("钻头专家评价", Arrays.asList("钻头专家", "专家评价", "评价", "钻头评价"));
        MENU_KEYWORDS.put("单井查询", Arrays.asList("单井", "单井查询"));
        MENU_KEYWORDS.put("分段钻参分析", Arrays.asList("分段钻参", "钻参分段", "分段", "分段钻井参数"));
        MENU_KEYWORDS.put("开次时间", Arrays.asList("开次时间", "开次", "钻井开次"));
        MENU_KEYWORDS.put("高指标井查询", Arrays.asList("高指标井", "高指标", "指标井查询"));
        MENU_KEYWORDS.put("典型地质岩性", Arrays.asList("地质岩性", "岩性", "典型岩性"));
        MENU_KEYWORDS.put("钻井液指标分析", Arrays.asList("钻井液指标", "泥浆指标","泥浆","钻井液", "钻井液分析"));
        MENU_KEYWORDS.put("造斜规律分析", Arrays.asList("造斜规律", "造斜分析", "造斜", "井斜规律", "井斜"));
    }

    // 井号匹配的正则表达式（匹配汉字+数字+特殊字符的井号格式）
    private static final String WELL_PATTERN_STR = "([\\u4e00-\\u9fa5]+[0-9]+[-\\u4e00-\\u9fa5]*[0-9]*)";
    private static final Pattern WELL_PATTERN = Pattern.compile(WELL_PATTERN_STR);

    /**
     * 识别用户语句中的菜单和井号
     * @param text 用户输入的自然语言语句
     * @return 识别结果（包含菜单名和井号）
     */
    public static RecognizeResult recognize(String text) {
        if (text == null || text.trim().isEmpty()) {
            return new RecognizeResult("", "");
        }
        String lowerText = text.toLowerCase(); // 转小写提高匹配容错率

        // 1. 提取井号
        String wellNumber = extractWellNumber(text);

        // 2. 匹配目标菜单
        String targetMenu = matchTargetMenu(lowerText);
        String uselessKeywords = "分析|查看|核对|查询|调取|查阅";
        if(StringUtils.isNotEmpty(wellNumber)){
            wellNumber=wellNumber.replaceAll(uselessKeywords, "");
        }
        return new RecognizeResult(targetMenu, wellNumber.trim());
    }

    /**
     * 从文本中提取井号
     * @param text 原始文本
     * @return 提取到的井号（无则返回空字符串）
     */
    private static String extractWellNumber(String text) {
        Matcher matcher = WELL_PATTERN.matcher(text);
        if (matcher.find()) {
            return matcher.group(1); // 返回第一个匹配的井号
        }
        return "";
    }

    /**
     * 匹配目标菜单
     * @param lowerText 小写后的用户文本
     * @return 匹配到的菜单名（无则返回空字符串）
     */
    private static String matchTargetMenu(String lowerText) {
        for (Map.Entry<String, List<String>> entry : MENU_KEYWORDS.entrySet()) {
            String menu = entry.getKey();
            List<String> keywords = entry.getValue();
            // 遍历当前菜单的所有关键词
            for (String keyword : keywords) {
                if (lowerText.contains(keyword.toLowerCase())) {
                    return menu; // 匹配到立即返回，保证效率
                }
            }
        }
        return "";
    }

    /**
     * 识别结果封装类
     * 用于返回菜单名和井号
     */
    public static class RecognizeResult {
        private String menu; // 目标菜单名
        private String wellNumber; // 井号

        public RecognizeResult(String menu, String wellNumber) {
            this.menu = menu;
            this.wellNumber = wellNumber;
        }

        // Getter方法
        public String getMenu() {
            return menu;
        }

        public String getWellNumber() {
            return wellNumber;
        }

        // 重写toString方便打印结果
        @Override
        public String toString() {
            return "识别结果：菜单=" + menu + "，井号=" + wellNumber;
        }
    }

    // 测试主方法
    public static void main(String[] args) {
        // 测试用例1：查询周期数据
        String testText1 = "我需要查看梁217-斜12的周期数据";
        RecognizeResult result1 = recognize(testText1);
        System.out.println("输入语句：" + testText1);
        System.out.println(result1 + "\n");

        // 测试用例2：查询邻井信息
        String testText2 = "帮我查询一下胜301的邻井数据";
        RecognizeResult result2 = recognize(testText2);
        System.out.println("输入语句：" + testText2);
        System.out.println(result2 + "\n");

        // 测试用例3：分析造斜规律
        String testText3 = "分析一下东58-斜8的造斜规律";
        RecognizeResult result3 = recognize(testText3);
        System.out.println("输入语句：" + testText3);
        System.out.println(result3);
    }
}