package com.zjsgfa.project.zjsgfa.util;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
public class FileBase64Converter {


    /**
     * 将文件转换为Base64编码字符串
     * @param file 要转换的文件对象
     * @return Base64编码字符串
     * @throws IOException 读取文件时的IO异常
     */
    public static String fileToBase64(File file) throws IOException {
        // 1. 校验文件是否有效
        if (file == null || !file.exists() || !file.isFile()) {
            throw new IllegalArgumentException("文件不存在或不是有效文件");
        }

        // 2. 读取文件字节
        try (FileInputStream fis = new FileInputStream(file)) {
            byte[] fileBytes = new byte[(int) file.length()];
            fis.read(fileBytes);

            // 3. 编码为Base64字符串（使用URL安全的编码器，避免+、/等特殊字符问题）
            return Base64.getUrlEncoder().encodeToString(fileBytes);
        }
    }

    /**
     * 将Base64字符串转回文件（用于测试验证）
     * @param base64Str Base64编码字符串
     * @param outputFile 输出文件对象
     * @throws IOException 写入文件时的IO异常
     */
    public static void base64ToFile(String base64Str, File outputFile) throws IOException {
        // 解码Base64字符串为字节数组
        byte[] fileBytes = Base64.getUrlDecoder().decode(base64Str);

        // 写入文件
        try (FileOutputStream fos = new FileOutputStream(outputFile)) {
            fos.write(fileBytes);
        }
    }


    /**
     * 将 Base64 字符串转换为 Excel 文件
     * @param base64Str 有效的 Excel 文件 Base64 编码字符串
     * @param outputPath 生成 Excel 文件的完整路径（如：D:\\test.xlsx）
     * @throws IOException 文件写入异常
     * @throws IllegalArgumentException Base64 字符串为空/无效
     */
    public static void base64ToExcel(String base64Str, String outputPath) throws IOException {
        // 1. 校验入参有效性
        if (base64Str == null || base64Str.isEmpty()) {
            throw new IllegalArgumentException("Base64 字符串不能为空");
        }
        if (outputPath == null || outputPath.isEmpty()) {
            throw new IllegalArgumentException("Excel 文件输出路径不能为空");
        }

        File excelFile = new File(outputPath);
        // 确保输出目录存在（如果目录不存在则创建）
        File parentDir = excelFile.getParentFile();
        if (!parentDir.exists()) {
            boolean mkdirsSuccess = parentDir.mkdirs();
            if (!mkdirsSuccess) {
                throw new IOException("创建输出目录失败：" + parentDir.getAbsolutePath());
            }
        }

        try {
            // 2. 解码 Base64 字符串为字节数组（兼容 URL 安全的 Base64 编码）
            byte[] excelBytes = Base64.getUrlDecoder().decode(base64Str);

            // 3. 将字节数组写入 Excel 文件（try-with-resources 自动关闭流）
            try (FileOutputStream fos = new FileOutputStream(excelFile)) {
                fos.write(excelBytes);
                System.out.println("Excel 文件生成成功：" + excelFile.getAbsolutePath());
            }
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Base64 字符串格式无效，无法解码", e);
        }
    }


    public static void main(String[] args) throws IOException {

            // 测试将文件转换为 Base64
        File file = new File("D:\\qianhe\\file\\钻井施工智能优选\\解析\\XYL217X52钻井地质设计.pdf");
        String base64Str = fileToBase64(file);
        System.out.println("Base64 编码字符串：" + base64Str);

        base64ToExcel(base64Str, "D:\\qianhe\\file\\钻井施工智能优选\\解析\\XYL217X52.pdf");
    }
}
