package com.qianhe.web.controller.tool;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class FileModificationChecker {

    public static void main(String[] args) {
        // 示例：查询2025年7月1日之后修改的文件
        LocalDateTime dateTime = LocalDateTime.of(2025, 7, 18, 0, 0);

        // 直接指定路径，不需要通过系统属性
        Path directory = Paths.get("D:\\SoftWork\\临时文件夹");

        try {
            List<Path> modifiedFiles = findFilesModifiedAfter(directory, dateTime);
            System.out.println("找到 " + modifiedFiles.size() + " 个在 " + dateTime + " 之后修改的文件：");
            for (Path file : modifiedFiles) {
                System.out.println(file);
            }
        } catch (IOException e) {
            System.err.println("扫描文件时出错：" + e.getMessage());
        }
    }

    /**
     * 查询指定目录下在某个日期之后修改的所有文件
     * @param directory 要扫描的目录
     * @param dateTime  指定日期时间
     * @return 符合条件的文件路径列表
     * @throws IOException 如果扫描过程中发生错误
     */
    public static List<Path> findFilesModifiedAfter(Path directory, LocalDateTime dateTime) throws IOException {
        List<Path> result = new ArrayList<>();
        long timeMillis = dateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();

        Files.walkFileTree(directory, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                // 检查文件修改时间
                if (attrs.lastModifiedTime().toMillis() > timeMillis) {
                    result.add(file);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) {
                System.err.println("无法访问文件: " + file + " - " + exc.getMessage());
                return FileVisitResult.CONTINUE;
            }
        });

        return result;
    }

    // 重载方法，支持使用java.util.Date作为参数
    public static List<Path> findFilesModifiedAfter(Path directory, Date date) throws IOException {
        LocalDateTime dateTime = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
        return findFilesModifiedAfter(directory, dateTime);
    }
}
