package com.qianhe.service.impl;

import com.qianhe.common.annotation.DataScope;
import com.qianhe.common.core.domain.entity.SysDept;
import com.qianhe.common.utils.DateUtils;
import com.qianhe.common.utils.SecurityUtils;
import com.qianhe.common.utils.StringUtils;
import com.qianhe.domain.*;
import com.qianhe.domain.Vo.SjfybpVo;
import com.qianhe.mapper.SjFybpMapper;
import com.qianhe.mapper.SjFybpWhMapper;
import com.qianhe.mapper.WdWdxxMapper;
import com.qianhe.service.IAmainService;
import com.qianhe.service.ISjFybpService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 三基-费用报批Service业务层处理
 *
 * @author qianhe
 * @date 2024-07-05
 */
@Service
public class AmainServiceImpl implements IAmainService
{

    /**
     * 查询例会查询情况
     */
    public  Map<String, Map<String, Integer>> statisticsMeetings(
            List<SysDept> depts, List<SjLhsc> lhList) {

        // 初始化结果Map，确保结构完整
        Map<String, Map<String, Integer>> result = new LinkedHashMap<>();
        result.put("Q1", new HashMap() {{ put("completed", 0); put("pending", 0); }});
        result.put("Q2", new HashMap() {{ put("completed", 0); put("pending", 0); }});
        result.put("Q3", new HashMap() {{ put("completed", 0); put("pending", 0); }});
        result.put("Q4", new HashMap() {{ put("completed", 0); put("pending", 0); }});

        // 处理单位列表为null的情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        // 获取所有单位ID，如果没有单位则返回全0的结果
        Set<String> allDeptIds = safeDepts.stream()
                .map(dept -> dept.getDeptId().toString())
                .collect(Collectors.toSet());

        if (allDeptIds.isEmpty()) {
            return result; // 没有单位时，所有季度的完成和未完成数都是0
        }

        // 处理例会列表为null的情况，转为空列表
        List<SjLhsc> safeLhList = (lhList == null) ? Collections.emptyList() : lhList;

        // 按季度分组统计已开例会的单位
        Map<String, Set<String>> quarterlyCompletedDepts = new HashMap<>();
        quarterlyCompletedDepts.put("Q1", new HashSet<>());
        quarterlyCompletedDepts.put("Q2", new HashSet<>());
        quarterlyCompletedDepts.put("Q3", new HashSet<>());
        quarterlyCompletedDepts.put("Q4", new HashSet<>());

        // 处理例会数据
        Calendar calendar = Calendar.getInstance();
        for (SjLhsc meeting : safeLhList) {
            // 跳过null的例会记录（如果列表中存在）
            if (meeting == null) continue;

            // 确定季度
            Date meetingDate = meeting.getHyrq();
            calendar.setTime(meetingDate);
            int month = calendar.get(Calendar.MONTH) + 1; // 月份从0开始，+1转为1-12
            String quarter = getQuarter(month);

            // 处理部门ID为null的情况
            String deptId = meeting.getDeptId();
            if (deptId != null && allDeptIds.contains(deptId)) {
                quarterlyCompletedDepts.get(quarter).add(deptId);
            }
        }

        // 计算每个季度的完成和未完成数量
        for (String quarter : result.keySet()) {
            Set<String> completedDepts = quarterlyCompletedDepts.get(quarter);
            int completed = completedDepts.size();
            int pending = allDeptIds.size() - completed;

            result.get(quarter).put("completed", completed);
            result.get(quarter).put("pending", pending);
        }

        return result;
    }


    /**
     * 统计各单位2025年各季度例会提交情况
     * @param depts 单位信息列表（可能为null）
     * @param lhList 2025年例会信息列表（可能为null）
     * @return 各单位季度提交状态列表
     */
    public  List<AmainXq> getDeptQuarterlyStatus(
            List<SysDept> depts, List<SjLhsc> lhList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjLhsc> safeLhList = (lhList == null) ? Collections.emptyList() : lhList;


        // 初始化所有单位的季度状态（默认未提交）
        Map<Long, AmainXq> deptStatusMap = safeDepts.stream()
                .map(dept -> new AmainXq(dept.getDeptId(),dept.getDeptName()))
                .collect(Collectors.toMap(AmainXq::getDeptId, status -> status));

        // 处理例会数据，标记已提交的季度
        Calendar calendar = Calendar.getInstance();
        for (SjLhsc meeting : safeLhList) {
            // 跳过null记录
            if (meeting == null) continue;

            // 获取会议日期和单位ID
            Date meetingDate = meeting.getHyrq();
            String deptIdStr = meeting.getDeptId();
            if (meetingDate == null || deptIdStr == null) continue;

            // 解析单位ID（字符串转Long）
            Long deptId;
            try {
                deptId = Long.parseLong(deptIdStr);
            } catch (NumberFormatException e) {
                // 单位ID格式错误，跳过该记录
                continue;
            }

            // 检查单位是否存在
            if (!deptStatusMap.containsKey(deptId)) {
                continue;
            }

            // 检查是否为2025年的会议
            calendar.setTime(meetingDate);
            int year = calendar.get(Calendar.YEAR);
            if (year != 2025) continue;

            // 确定季度
            int month = calendar.get(Calendar.MONTH) + 1; // 月份从0开始
            String quarter = getQuarter(month);

            // 更新对应季度的状态为"已提交"
            AmainXq status = deptStatusMap.get(deptId);
            switch (quarter) {
                case "Q1":
                    status.setQ1("已提交");
                    break;
                case "Q2":
                    status.setQ2("已提交");
                    break;
                case "Q3":
                    status.setQ3("已提交");
                    break;
                case "Q4":
                    status.setQ4("已提交");
                    break;
            }
        }

        // 转换为列表并按单位ID排序
        return deptStatusMap.values().stream()
                .sorted(Comparator.comparing(AmainXq::getDeptId))
                .collect(Collectors.toList());

    }



        /**
         * 根据月份获取季度
         */
    private static String getQuarter(int month) {
        if (month >= 1 && month <= 3) {
            return "Q1";
        } else if (month >= 4 && month <= 6) {
            return "Q2";
        } else if (month >= 7 && month <= 9) {
            return "Q3";
        } else {
            return "Q4";
        }
    }

    /**
     * 查询费用提交情况
     */
    public  Map countDepartmentExpenses(List<SysDept> depts, List<SjFybpZb> fyList) {

        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjFybpZb> safeFyList = (fyList == null) ? Collections.emptyList() : fyList;

        // 获取所有单位ID
        Set<String> allDeptIds = safeDepts.stream()
                .map(dept -> dept.getDeptId().toString())
                .collect(Collectors.toSet());

        // 总单位数
        int totalDepts = allDeptIds.size();

        // 获取有费用信息的单位ID（去重且有效）
        Set<String> deptsWithExpense = safeFyList.stream()
                .filter(fy -> fy != null && fy.getDeptId() != null) // 过滤无效记录
                .map(SjFybpZb::getDeptId)
                .filter(allDeptIds::contains) // 只保留存在于单位列表中的ID
                .collect(Collectors.toSet());

        // 计算数量
        int completed = deptsWithExpense.size();
        int pending = totalDepts - completed;

        // 整理结果
        Map<String, Integer> result = new HashMap<>();
        result.put("completed", completed);
        result.put("pending", pending);
        return result;
    }


    /**
     * 统计各单位费用提交情况
     * @param depts 单位信息列表（可能为null）
     * @param fyList 费用信息列表（可能为null）
     * @return 各单位费用提交状态列表
     */
    public  List<AmainXq> getExpenseStatus(
            List<SysDept> depts, List<SjFybpZb> fyList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjFybpZb> safeFyList = (fyList == null) ? Collections.emptyList() : fyList;

        // 初始化所有单位的状态（默认未提交）
        Map<Long, AmainXq> statusMap = safeDepts.stream()
                .map(dept -> new AmainXq(dept.getDeptId(),dept.getDeptName()))
                .collect(Collectors.toMap(
                        AmainXq::getDeptId,
                        status -> status,
                        (existing, replacement) -> existing // 处理重复deptId
                ));

        // 提取所有有费用信息的单位ID（去重）
        Set<Long> hasExpenseDeptIds = safeFyList.stream()
                .filter(expense -> expense != null && expense.getDeptId() != null)
                .map(expense -> {
                    // 尝试将字符串deptId转换为Long
                    try {
                        return Long.parseLong(expense.getDeptId());
                    } catch (NumberFormatException e) {
                        return null; // 转换失败的ID忽略
                    }
                })
                .filter(Objects::nonNull) // 过滤转换失败的null值
                .filter(statusMap::containsKey) // 只保留存在于单位列表中的ID
                .collect(Collectors.toSet());

        // 更新有费用信息的单位状态为"已提交"
        for (Long deptId : hasExpenseDeptIds) {
            statusMap.get(deptId).setQ1("已提交");
        }

        // 转换为列表并按单位ID排序
        return statusMap.values().stream()
                .sorted(Comparator.comparing(AmainXq::getDeptId))
                .collect(Collectors.toList());
    }

    /**
     * 按上半年和下半年统计各单位分档评价提交情况
     * @param depts 单位信息列表（可能为null）
     * @param fdpjList 分档评价信息列表（可能为null）
     * @return 各单位分档评价提交状态列表
     */
    public  Map<String, Map<String, Integer>> statisticsGradingEvaluation(
            List<SysDept> depts, List<SjGfjsZsdwkhpjZb> fdpjList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjGfjsZsdwkhpjZb> safeFdpjList = (fdpjList == null) ? Collections.emptyList() : fdpjList;

        // 获取所有单位ID（Long类型）
        Set<Long> allDeptIds = safeDepts.stream()
                .map(SysDept::getDeptId)
                .collect(Collectors.toSet());

        // 总单位数
        int totalDepts = allDeptIds.size();

        // 初始化各周期已提交单位的集合
        Set<Long> firstHalfSubmitted = new HashSet<>();
        Set<Long> secondHalfSubmitted = new HashSet<>();

        // 处理分档评价数据
        for (SjGfjsZsdwkhpjZb evaluation : safeFdpjList) {
            // 跳过null的评价记录
            if (evaluation == null) continue;

            Long deptId = evaluation.getDeptId();
            String period = evaluation.getZqfw();

            // 过滤无效数据（部门ID为null、周期为空、不在单位列表中）
            if (deptId == null || period == null || !allDeptIds.contains(deptId)) {
                continue;
            }

            // 按周期分类
            if ("上半年".equals(period)) {
                firstHalfSubmitted.add(deptId);
            } else if ("全年".equals(period)) {
                secondHalfSubmitted.add(deptId);
            }
        }

        // 计算各周期的统计结果
        Map<String, Integer> firstHalfStats = new HashMap<>();
        firstHalfStats.put("completed", firstHalfSubmitted.size());
        firstHalfStats.put("pending", totalDepts - firstHalfSubmitted.size());

        Map<String, Integer> secondHalfStats = new HashMap<>();
        secondHalfStats.put("completed", secondHalfSubmitted.size());
        secondHalfStats.put("pending", totalDepts - secondHalfSubmitted.size());

        // 整理最终结果
        Map<String, Map<String, Integer>> result = new HashMap<>();
        result.put("Q1", firstHalfStats);
        result.put("Q2", secondHalfStats);

        return result;
    }

    /**
     * 按上半年和下半年统计各单位分档评价提交情况
     * @param depts 单位信息列表（可能为null）
     * @param fdpjList 分档评价信息列表（可能为null）
     * @return 各单位分档评价提交状态列表
     */
    public  List<AmainXq> getEvaluationStatus(
            List<SysDept> depts, List<SjGfjsZsdwkhpjZb> fdpjList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjGfjsZsdwkhpjZb> safeFdpjList = (fdpjList == null) ? Collections.emptyList() : fdpjList;

        // 初始化所有单位的状态（默认未提交）
        Map<Long, AmainXq> statusMap = safeDepts.stream()
                .map(dept -> new AmainXq(dept.getDeptId(),dept.getDeptName()))
                .collect(Collectors.toMap(
                        AmainXq::getDeptId,
                        status -> status,
                        (existing, replacement) -> existing // 处理重复deptId
                ));

        // 处理分档评价数据，更新对应周期的状态
        for (SjGfjsZsdwkhpjZb evaluation : safeFdpjList) {
            // 跳过null记录
            if (evaluation == null) continue;

            Long deptId = evaluation.getDeptId();
            String period = evaluation.getZqfw();

            // 过滤无效数据
            if (deptId == null || period == null || !statusMap.containsKey(deptId)) {
                continue;
            }

            // 更新对应周期的状态为"已提交"
            AmainXq status = statusMap.get(deptId);
            if ("上半年".equals(period)) {
                status.setQ1("已提交");
            } else if ("全年".equals(period)) {
                status.setQ2("已提交");
            }
        }

        // 转换为列表并按单位ID排序
        return statusMap.values().stream()
                .sorted(Comparator.comparing(AmainXq::getDeptId))
                .collect(Collectors.toList());
    }


    /**
     * 统计每个月的月度问题记录填写情况（使用数字月份01-12）
     * @param depts 单位信息列表（可能为null）
     * @param ydwtList 月度问题记录列表（可能为null）
     * @return 包含各月份统计结果的Map
     */
    public  Map<String, Map<String, Integer>> statisticsMonthlyIssues(
            List<SysDept> depts, List<SjGfjsYdjcjl> ydwtList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjGfjsYdjcjl> safeYdwtList = (ydwtList == null) ? Collections.emptyList() : ydwtList;

        // 获取所有单位ID
        Set<Long> allDeptIds = safeDepts.stream()
                .map(SysDept::getDeptId)
                .collect(Collectors.toSet());

        // 获取单位总数
        int totalDepts = allDeptIds.size();

        // 初始化各月份（01-12）已提交单位的集合
        Map<String, Set<Long>> monthlySubmitted = new LinkedHashMap<>();
        // 按顺序添加12个月份
        for (int i = 1; i <= 12; i++) {
            String month = String.format("%02d", i); // 格式化为两位数字（01-12）
            monthlySubmitted.put(month, new HashSet<>());
        }

        // 处理月度问题记录数据
        for (SjGfjsYdjcjl record : safeYdwtList) {
            // 跳过null的记录
            if (record == null) continue;

            Long deptId = record.getDeptId();
            String dateStr = record.getRq();

            // 过滤无效数据
            if (deptId == null || dateStr == null || !allDeptIds.contains(deptId)) {
                continue;
            }

            // 解析月份（格式为"yyyy-MM"）
            String[] dateParts = dateStr.split("-");
            if (dateParts.length != 2) {
                continue; // 日期格式不正确，跳过
            }
            String month = dateParts[1]; // 获取月份部分（如"07"）

            // 如果是有效月份（01-12），添加到对应集合
            if (monthlySubmitted.containsKey(month)) {
                monthlySubmitted.get(month).add(deptId);
            }
        }

        // 计算各月份的统计结果
        Map<String, Map<String, Integer>> result = new LinkedHashMap<>();
        for (String month : monthlySubmitted.keySet()) {
            Set<Long> submittedDepts = monthlySubmitted.get(month);

            Map<String, Integer> monthStats = new HashMap<>();
            monthStats.put("completed", submittedDepts.size());
            monthStats.put("pending", totalDepts - submittedDepts.size());

            result.put(month, monthStats);
        }

        return result;
    }


    /**
     * 按月份统计各单位月度问题提交情况
     * @param depts 单位信息列表（可能为null）
     * @param ydwtList 月度问题记录列表（可能为null）
     * @return 各单位月度问题提交状态列表
     */
    public  List<AmainXq> getMonthlyIssueStatus(
            List<SysDept> depts, List<SjGfjsYdjcjl> ydwtList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjGfjsYdjcjl> safeYdwtList = (ydwtList == null) ? Collections.emptyList() : ydwtList;

        // 初始化所有单位的月度状态（默认未提交）
        Map<Long, AmainXq> statusMap = safeDepts.stream()
                .map(dept -> new AmainXq(dept.getDeptId(),dept.getDeptName()))
                .collect(Collectors.toMap(
                        AmainXq::getDeptId,
                        status -> status,
                        (existing, replacement) -> existing // 处理重复deptId
                ));

        // 处理月度问题记录数据
        for (SjGfjsYdjcjl record : safeYdwtList) {
            // 跳过null的记录
            if (record == null) continue;

            Long deptId = record.getDeptId();
            String dateStr = record.getRq();

            // 过滤无效数据
            if (deptId == null || dateStr == null || !statusMap.containsKey(deptId)) {
                continue;
            }

            // 解析月份（格式为"yyyy-MM"）
            String[] dateParts = dateStr.split("-");
            if (dateParts.length != 2) {
                continue; // 日期格式不正确，跳过
            }
            String month = dateParts[1]; // 获取月份部分（如"07"）

            // 根据月份更新对应状态为"已提交"
            AmainXq status = statusMap.get(deptId);
            switch (month) {
                case "01":
                    status.setY01("已提交");
                    break;
                case "02":
                    status.setY02("已提交");
                    break;
                case "03":
                    status.setY03("已提交");
                    break;
                case "04":
                    status.setY04("已提交");
                    break;
                case "05":
                    status.setY05("已提交");
                    break;
                case "06":
                    status.setY06("已提交");
                    break;
                case "07":
                    status.setY07("已提交");
                    break;
                case "08":
                    status.setY08("已提交");
                    break;
                case "09":
                    status.setY09("已提交");
                    break;
                case "10":
                    status.setY10("已提交");
                    break;
                case "11":
                    status.setY11("已提交");
                    break;
                case "12":
                    status.setY12("已提交");
                    break;
            }
        }

        // 转换为列表并按单位ID排序
        return statusMap.values().stream()
                .sorted(Comparator.comparing(AmainXq::getDeptId))
                .collect(Collectors.toList());
    }


    public  Map<String, Map<String, Integer>> statisticsQuarterlyTests(
            List<SysDept> depts, List<SjGfjsZsdwkhpjZb> fdpjList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjGfjsZsdwkhpjZb> safeFdpjList = (fdpjList == null) ? Collections.emptyList() : fdpjList;

        // 获取所有单位ID
        Set<Long> allDeptIds = safeDepts.stream()
                .map(SysDept::getDeptId)
                .collect(Collectors.toSet());
        int totalDepts = allDeptIds.size();

        // 季度名称映射（中文到Q1-Q4）
        Map<String, String> quarterMap = new HashMap<>();
        quarterMap.put("第一季度", "Q1");
        quarterMap.put("第二季度", "Q2");
        quarterMap.put("第三季度", "Q3");
        quarterMap.put("第四季度", "Q4");

        // 初始化各季度已完成单位的集合
        Map<String, Set<Long>> quarterlyCompleted = new LinkedHashMap<>();
        quarterlyCompleted.put("Q1", new HashSet<>());
        quarterlyCompleted.put("Q2", new HashSet<>());
        quarterlyCompleted.put("Q3", new HashSet<>());
        quarterlyCompleted.put("Q4", new HashSet<>());

        // 处理季度自测数据
        for (SjGfjsZsdwkhpjZb record : safeFdpjList) {
            // 跳过null的记录
            if (record == null) continue;

            Long deptId = record.getDeptId();
            String period = record.getZqfw();

            // 过滤无效数据
            if (deptId == null || period == null || !allDeptIds.contains(deptId)) {
                continue;
            }

            // 转换为Q1-Q4格式
            String quarter = quarterMap.get(period);
            if (quarter != null) {
                quarterlyCompleted.get(quarter).add(deptId);
            }
        }

        // 计算各季度的统计结果
        Map<String, Map<String, Integer>> result = new LinkedHashMap<>();
        for (String quarter : quarterlyCompleted.keySet()) {
            Set<Long> completedDepts = quarterlyCompleted.get(quarter);

            Map<String, Integer> quarterStats = new HashMap<>();
            quarterStats.put("completed", completedDepts.size());
            quarterStats.put("pending", totalDepts - completedDepts.size());

            result.put(quarter, quarterStats);
        }

        return result;
    }


    /**
     * 按季度统计各单位自测提交情况
     * @param depts 单位信息列表（可能为null）
     * @param fdpjList 季度自测记录列表（可能为null）
     * @return 各单位季度自测提交状态列表
     */
    public  List<AmainXq> getQuarterlyTestStatus(
            List<SysDept> depts, List<SjGfjsZsdwkhpjZb> fdpjList) {

        // 处理null情况，转为空列表
        List<SysDept> safeDepts = (depts == null) ? Collections.emptyList() : depts;
        List<SjGfjsZsdwkhpjZb> safeFdpjList = (fdpjList == null) ? Collections.emptyList() : fdpjList;

        // 初始化所有单位的季度状态（默认未提交）
        Map<Long, AmainXq> statusMap = safeDepts.stream()
                .map(dept -> new AmainXq(dept.getDeptId(),dept.getDeptName()))
                .collect(Collectors.toMap(
                        AmainXq::getDeptId,
                        status -> status,
                        (existing, replacement) -> existing // 处理重复deptId
                ));

        // 处理季度自测数据，更新对应季度的状态
        for (SjGfjsZsdwkhpjZb testRecord : safeFdpjList) {
            // 跳过null记录
            if (testRecord == null) continue;

            Long deptId = testRecord.getDeptId();
            String period = testRecord.getZqfw();

            // 过滤无效数据
            if (deptId == null || period == null || !statusMap.containsKey(deptId)) {
                continue;
            }

            // 更新对应季度的状态为"已提交"
            AmainXq status = statusMap.get(deptId);
            switch (period) {
                case "第一季度":
                    status.setQ1("已提交");
                    break;
                case "第二季度":
                    status.setQ2("已提交");
                    break;
                case "第三季度":
                    status.setQ3("已提交");
                    break;
                case "第四季度":
                    status.setQ4("已提交");
                    break;
            }
        }

        // 转换为列表并按单位ID排序
        return statusMap.values().stream()
                .sorted(Comparator.comparing(AmainXq::getDeptId))
                .collect(Collectors.toList());
    }



}