package com.ruoyi.project.ys.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import com.ruoyi.common.utils.DateUtils;
import com.ruoyi.framework.config.RuoYiConfig;
import com.ruoyi.project.ys.domain.HyjyxxFile;
import com.ruoyi.project.ys.mapper.HyjyxxFileMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.project.ys.mapper.HyjyxxMapper;
import com.ruoyi.project.ys.domain.Hyjyxx;
import com.ruoyi.project.ys.service.IHyjyxxService;


import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.ZoneId;

import static com.ruoyi.common.utils.SecurityUtils.getUserId;

/**
 * 会议信息Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-09-17
 */
@Service
public class HyjyxxServiceImpl implements IHyjyxxService 
{
    @Autowired
    private HyjyxxMapper hyjyxxMapper;

    /**
     * 查询会议信息
     * 
     * @param id 会议信息主键
     * @return 会议信息
     */
    @Override
    public Hyjyxx selectHyjyxxById(Long id)
    {
        return hyjyxxMapper.selectHyjyxxById(id);
    }

    /**
     * 查询会议信息列表
     * 
     * @param hyjyxx 会议信息
     * @return 会议信息
     */
    @Override
    public List<Hyjyxx> selectHyjyxxList(Hyjyxx hyjyxx)
    {
        return hyjyxxMapper.selectHyjyxxList(hyjyxx);
    }

    /**
     * 新增会议信息
     * 
     * @param hyjyxx 会议信息
     * @return 结果
     */
    @Override
    public int insertHyjyxx(Hyjyxx hyjyxx)
    {
        hyjyxx.setCreatedBy(getUserId()+"");
        hyjyxx.setCreatedTime(DateUtils.getNowDate());
        return hyjyxxMapper.insertHyjyxx(hyjyxx);
    }

    /**
     * 修改会议信息
     * 
     * @param hyjyxx 会议信息
     * @return 结果
     */
    @Override
    public int updateHyjyxx(Hyjyxx hyjyxx)
    {
        hyjyxx.setUpdateBy(getUserId()+"");
        hyjyxx.setUpdateTime(DateUtils.getNowDate());
        return hyjyxxMapper.updateHyjyxx(hyjyxx);
    }

    /**
     * 批量删除会议信息
     * 
     * @param ids 需要删除的会议信息主键
     * @return 结果
     */
    @Override
    public int deleteHyjyxxByIds(Long[] ids)
    {
        return hyjyxxMapper.deleteHyjyxxByIds(ids);
    }

    /**
     * 删除会议信息信息
     * 
     * @param id 会议信息主键
     * @return 结果
     */
    @Override
    public int deleteHyjyxxById(Long id)
    {
        return hyjyxxMapper.deleteHyjyxxById(id);
    }


    // 1. 转录接口地址（固定）
    private static final String ASR_API_URL = "/asr";
    /**
     * 修改会议信息
     *
     * @param id 会议信息
     * @return 结果
     */
    @Override
    public int hyzl(Long id)
    {
        Hyjyxx hyjyxx =hyjyxxMapper.selectHyjyxxById(id);
        String targetMp3Url = RuoYiConfig.getDzysip() + hyjyxx.getYpwj();
        try {
            // 1. 发送 POST 请求，获取原始 JSON 响应
            String responseJson = sendAsrPostRequest(targetMp3Url);
            System.out.println("接口返回原始 JSON：" + responseJson);

            // 创建ObjectMapper实例
            ObjectMapper objectMapper = new ObjectMapper();

            // 将String转换为JsonNode（JSON对象）
            JsonNode jsonNode = objectMapper.readTree(responseJson);

            // 2. 直接提取 asr_text 和 timestamp_text 的值
            String asrText = jsonNode.get("asr_text").asText();
            String timestampText = jsonNode.get("json_result").toString();
            hyjyxx.setYw(asrText);
            hyjyxx.setYwBzh(timestampText);
            hyjyxx.setJxzt("转录成功");
            return hyjyxxMapper.updateHyjyxx(hyjyxx);
        } catch (IOException e) {
            System.err.println("请求异常：" + e.getMessage());
            e.printStackTrace();
            hyjyxx.setJxzt("转录失败");
            hyjyxxMapper.updateHyjyxx(hyjyxx);
            return 0;
        }
    }


    /**
     * 发送 POST 请求到 ASR 接口
     * @param mp3Url 音频文件 URL（对应请求参数 mp3_url）
     * @return 接口返回结果（Map 格式：key=asr_text/timestamp_text，value=对应文本）
     * @throws IOException 网络异常/请求失败
     */
    public static String sendAsrPostRequest(String mp3Url) throws IOException {
        // 1. 构建请求 JSON 字符串（无实体类，直接拼接）
        String requestJson = String.format("{\"mp3_url\":\"%s\"}", mp3Url);

        // 2. 创建 URL 对象并打开连接
        URL url = new URL(RuoYiConfig.getYpzlip()+ASR_API_URL);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();

        try {
            // 3. 配置请求参数（POST 方法、请求头、超时时间）
            connection.setRequestMethod("POST"); // 必须设为 POST
            connection.setRequestProperty("Content-Type", "application/json; charset=UTF-8"); // 声明 JSON 格式
            connection.setDoOutput(true); // 允许向服务器写数据（POST 必须开启）
            connection.setConnectTimeout(5000); // 连接超时：5秒
            connection.setReadTimeout(500000); // 读取响应超时：10秒

            // 4. 向服务器写入请求体（JSON 字符串）
            try (DataOutputStream outputStream = new DataOutputStream(connection.getOutputStream())) {
                byte[] requestBytes = requestJson.getBytes(StandardCharsets.UTF_8);
                outputStream.write(requestBytes); // 写入请求数据
                outputStream.flush(); // 刷新缓冲区，确保数据全部发送
            }

            // 5. 检查请求是否成功（HTTP 状态码 200-299 为成功）
            int responseCode = connection.getResponseCode();
            if (responseCode < 200 || responseCode >= 300) {
                throw new IOException("请求失败！HTTP 状态码：" + responseCode
                        + "，响应信息：" + connection.getResponseMessage());
            }

            // 6. 读取接口返回的 JSON 响应
            StringBuilder responseSb = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(
                    new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8))) {
                String line;
                while ((line = reader.readLine()) != null) {
                    responseSb.append(line); // 逐行读取响应内容
                }
            }
            String responseJson = responseSb.toString();

            // 7. 将 JSON 转为 Map（无实体类，直接通过 key 取值）
            return responseJson;

        } finally {
            // 8. 关闭连接（避免资源泄漏）
            connection.disconnect();
        }
    }

    @Autowired
    private HyjyxxFileMapper hyjyxxFileMapper;

    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SS");
    // 修正：最后一个说话人的区间结束时间（用一个远大于实际可能的时间，等效于“无穷大”）
    private static final LocalDateTime MAX_TIME = LocalDateTime.of(2100, 1, 1, 0, 0, 0); // 2100年1月1日


    @Override
    public Hyjyxx selectHyjyxxById_xq(Long id)
    {
        Hyjyxx hyxx = hyjyxxMapper.selectHyjyxxById(id);
        HyjyxxFile file = new HyjyxxFile();
        file.setHyid(id);
        List<HyjyxxFile>  fileList = hyjyxxFileMapper.selectHyjyxxFileList(file);
        String originalJson = hyxx.getYwBzh();
        Date hysj =hyxx.getJxwcsj();


        try {
            // 初始化ObjectMapper
            ObjectMapper objectMapper = new ObjectMapper();
            // 解析JSON为ObjectNode列表（便于修改）
            List<ObjectNode> speakerList = objectMapper.readValue(
                    originalJson,
                    new TypeReference<List<ObjectNode>>() {}
            );

            // 解析录音开始时间
            LocalDateTime startTime = hysj.toInstant()
                    .atZone(ZoneId.of("Asia/Shanghai"))
                    .toLocalDateTime();

            // 遍历每个说话人，计算并添加真实时间
            for (ObjectNode speakerNode : speakerList) {
                // 获取timestamp字段
                String timestamp = speakerNode.get("timestamp").asText();
                // 计算真实时间
                LocalDateTime realTime = calculateRealTime(startTime, timestamp); // 这个人说话开始
                // 新增realTime字段
                speakerNode.put("realTime", realTime.format(TIME_FORMATTER));
            }

            // 遍历每个说话人，计算并添加真实时间
            for (int i = 0; i < speakerList.size(); i++){
                ObjectNode speakerNode= speakerList.get(i);
                String  realTime = speakerNode.get("realTime").asText();
                LocalDateTime currentStartTime = parseStringToLocalDateTime(realTime);
                LocalDateTime intervalEnd; // 区间结束时间

                // 构建区间：当前说话人开始时间 → 下一个说话人开始时间（最后一个说话人特殊处理）
                if (i < speakerList.size() - 1) {
                    // 非最后一个说话人：区间结束 = 下一个说话人开始时间
                    intervalEnd = parseStringToLocalDateTime(speakerList.get(i + 1).get("realTime").asText());
                } else {
                    // 最后一个说话人：区间结束 = 当前开始时间 + 延长时间
                    intervalEnd = MAX_TIME;
                }
                List<String> matchedScreenshots = new ArrayList<>();
                for (HyjyxxFile hyjyxxFile : fileList) {
                    Date fileTime = hyjyxxFile.getFileTime();
                    LocalDateTime screenTime   = fileTime.toInstant()
                            .atZone(ZoneId.of("Asia/Shanghai"))
                            .toLocalDateTime();
                    if (screenTime.isAfter(currentStartTime) || screenTime.isEqual(currentStartTime)) {
                        if (screenTime.isBefore(intervalEnd)) {
                            matchedScreenshots.add(hyjyxxFile.getFileUrl());
                        }
                    }
                }
                speakerNode.putPOJO("file",matchedScreenshots);
            }

            // 输出处理后的JSON（可写入文件或返回）
            String resultJson = objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(speakerList);
            System.out.println("处理后带真实时间的JSON：");
            System.out.println(resultJson);

            hyxx.setYwBzh(resultJson);

        } catch (Exception e) {
            e.printStackTrace();
        }
        return hyxx;
    }

    /**
     * 计算真实时间：开始时间 + timestamp偏移量
     */
    private static LocalDateTime calculateRealTime(LocalDateTime startTime, String timestamp) {
        try {
            // 拆分timestamp为 时:分:秒.毫秒（如"00:00:25.86" → 0时0分25秒86毫秒）
            String[] timeParts = timestamp.split(":");
            int hours = Integer.parseInt(timeParts[0]);
            int minutes = Integer.parseInt(timeParts[1]);
            String[] secAndMs = timeParts[2].split("\\.");
            int seconds = Integer.parseInt(secAndMs[0]);
            int millis = secAndMs.length > 1 ? Integer.parseInt(secAndMs[1]) : 0;

            // 计算总偏移量（毫秒）
            long totalMillis = hours * 3600000L + minutes * 60000L + seconds * 1000L + millis;
            // 开始时间 + 偏移量
            return startTime.plus(Duration.ofMillis(totalMillis));

        } catch (ArrayIndexOutOfBoundsException | NumberFormatException e) {
            throw new DateTimeParseException("解析timestamp失败：" + timestamp, timestamp, 0);
        }
    }

    /**
     * 工具方法：将String时间解析为LocalDateTime（处理解析异常）
     */
    private static LocalDateTime parseStringToLocalDateTime(String timeStr) {
        try {
            return LocalDateTime.parse(timeStr, TIME_FORMATTER);
        } catch (DateTimeParseException e) {
            System.err.println("时间解析失败：" + timeStr + "，格式需为" + TIME_FORMATTER);
            return null; // 解析失败返回null，后续跳过该数据
        }
    }



}
