package com.ruoyi.project.dz.utils;

import org.apache.commons.io.IOUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

/**
 * 读取segy文件
 * Created by wj on 2023/8/30.
 */
public class SegyReadUtil {

    //EBCDIC文件头（3200字节）
    private static int EBCDIC_FILE_HEADER_LENGTH = 3200;
    //二进制文件头（400字节）
    private static int BINARY_FILE_HEADER_LENGTH = 400;

    //数据道道头 道头（Trace Header）：长度：240 bytes；数据类型：32位、16位的整型；
    //这部分包含每个道数据的元数据信息，如道序列号、道标识代码、坐标位置和记录参数等。
    private static int TRACE_HEADER_LENGTH = 240;


    /**
     * EBCDIC文件头（3200字节） 由40个卡组成（例如：每行80个字符*40行），需要转换为ASCII码后才能显示。
     * 主要用于记录与整个工区相关的信息，如施工单位、施工区、监理单位等。
     **/
    public static SegyTextFileHeader getFileHeader3200 (String filePath) throws IOException {
        FileInputStream fis = null;
        DataInputStream dis = null;
        SegyTextFileHeader txtHdr = null;

        try {
            fis = new FileInputStream(filePath);
            dis = new DataInputStream(fis);

            // 首先读取前3200字节的文本头
            byte[] buf = new byte[EBCDIC_FILE_HEADER_LENGTH];
            dis.readFully(buf);
            byte[] b = CodeUtil.EBCDICToASCII(buf);
            txtHdr = new SegyTextFileHeader(b);

        } catch (IOException e) {
            System.out.println("读取文件头报错。");
            e.printStackTrace();
        } finally {
            IOUtils.close(fis);
            IOUtils.close(dis);
        }

        return txtHdr;
    }


    /**
     * 二进制文件头（400字节）
     * 用来存储描述SEG-Y文件的一些关键信息，包括SEG-Y文件的数据格式、采样点数、采样间隔等。
     **/
    public static SegyReelHdr getFileHeader400 (String filePath) throws IOException {
        FileInputStream fis = null;
        DataInputStream dis = null;
        SegyReelHdr reelHdr = null;

        try {
            fis = new FileInputStream(filePath);
            dis = new DataInputStream(fis);

            // 首先读取前3200字节的文本头
            byte[] buf = new byte[BINARY_FILE_HEADER_LENGTH];
            dis.skipBytes(EBCDIC_FILE_HEADER_LENGTH);//跳过 开头 3200 个字节
            dis.readFully(buf);
            reelHdr = new SegyReelHdr(buf);
            //判断是否有扩展卷头
            if(reelHdr.has_extend_text_hdr_flag!=0){
                //此时有3200字节的扩展原文文件头
                //需要执行读取动作，此处暂空，以抛出异常为处理手段
                System.out.println("存在扩展原文文件头，需要读取！");
            }

        } catch (IOException e) {
            System.out.println("读取文件头报错。");
            e.printStackTrace();
        } finally {
            IOUtils.close(fis);
            IOUtils.close(dis);
        }

        return reelHdr;
    }
    /**
     * 输出byte[]
     **/
    public static byte[] getFileHeaderByteArr400 (String filePath) throws IOException {
        FileInputStream fis = null;
        DataInputStream dis = null;
        byte[] buf = new byte[BINARY_FILE_HEADER_LENGTH];

        try {
            fis = new FileInputStream(filePath);
            dis = new DataInputStream(fis);

            dis.skipBytes(EBCDIC_FILE_HEADER_LENGTH);//跳过 开头 3200 个字节
            dis.readFully(buf);

        } catch (IOException e) {
            System.out.println("读取文件头报错。");
            e.printStackTrace();
        } finally {
            IOUtils.close(fis);
            IOUtils.close(dis);
        }

        return buf;
    }

    /**
     * 根据指定线号道号获取道数据
     * filePath 文件路径
     * xh 线号 野外原始记录号 xline
     * dh 道号 道集号 CDP等
     * txtHdr 文件头信息 包含 线和道的范围 最大最小等信息
     * reelHdr 卷头信息 包含 数据道采样点数 、 长度等信息
     **/
    public static SegyTrace getOneTraceByNo (String filePath, int xh, int dh, SegyTextFileHeader txtHdr,
                                             SegyReelHdr reelHdr) throws IOException {
        FileInputStream fis = null;
        DataInputStream dis = null;
        SegyTrace oneTrace = null;

        try {
            fis = new FileInputStream(filePath);
            dis = new DataInputStream(fis);

            //跳过文件头、卷头
            dis.skipBytes(EBCDIC_FILE_HEADER_LENGTH);//跳过 开头 3200 个字节
            dis.skipBytes(BINARY_FILE_HEADER_LENGTH);//跳过 卷头 400 个字节

            // 根据卷头得出每一块数据集的大小 240+每道采样点数*根据编码格式获取的数据长度=每一道字符长度
            int samples_per_trace = reelHdr.samples_per_trace;
            int sample_length = reelHdr.getMultiple();
            int dao_dataCount = TRACE_HEADER_LENGTH + reelHdr.samples_per_trace * reelHdr.getMultiple();

            //根据线号、道号 以及 线号、道号的最小最大值（即范围） 来计算需要跳过的道数量
            int xh_min = txtHdr.first_xline;
            int xh_max = txtHdr.last_xline;
            int dh_min = txtHdr.first_inline;
            int dh_max = txtHdr.last_inline;
            int skip_trace_num = (xh - xh_min) * (dh_max - dh_min + 1) + (dh - dh_min);//需要跳过的道数量
            int skip_trace_length = skip_trace_num * dao_dataCount;//需要跳过的道长度
            dis.skipBytes(skip_trace_length);//跳过

            //读取一道数据返回 (分开读取道头和数据)
            byte[] buf = new byte[TRACE_HEADER_LENGTH];
            dis.readFully(buf);
            oneTrace = new SegyTrace(buf);//道头
            buf = new byte[reelHdr.samples_per_trace * reelHdr.getMultiple()];
            dis.readFully(buf);
            oneTrace.readDataInIBMFormat(buf);


        } catch (IOException e) {
            System.out.println("读取文件头报错。");
            e.printStackTrace();
        } catch (SegyFileFormatException e) {
            System.out.println("读取卷头报错：数据编码格式及大小。");
            e.printStackTrace();
        } finally {
            IOUtils.close(fis);
            IOUtils.close(dis);
        }

        return oneTrace;
    }
    /**
     * 根据指定线号道号获取道数据 获取byte数组
     * filePath 文件路径
     * xh 线号 野外原始记录号 xline
     * dh 道号 道集号 CDP等
     * txtHdr 文件头信息 包含 线和道的范围 最大最小等信息
     * reelHdr 卷头信息 包含 数据道采样点数 、 长度等信息
     **/
    public static byte[] getOneTraceByteByNo (String filePath, int xh, int dh, SegyTextFileHeader txtHdr,
                                             SegyReelHdr reelHdr) throws IOException {
        FileInputStream fis = null;
        DataInputStream dis = null;
        SegyTrace oneTrace = null;
        byte[] buf = null;

        try {
            fis = new FileInputStream(filePath);
            dis = new DataInputStream(fis);

            //跳过文件头、卷头
            dis.skipBytes(EBCDIC_FILE_HEADER_LENGTH);//跳过 开头 3200 个字节
            dis.skipBytes(BINARY_FILE_HEADER_LENGTH);//跳过 卷头 400 个字节

            // 根据卷头得出每一块数据集的大小 240+每道采样点数*根据编码格式获取的数据长度=每一道字符长度
            int samples_per_trace = reelHdr.samples_per_trace;
            int sample_length = reelHdr.getMultiple();
            int dao_dataCount = TRACE_HEADER_LENGTH + reelHdr.samples_per_trace * reelHdr.getMultiple();

            //根据线号、道号 以及 线号、道号的最小最大值（即范围） 来计算需要跳过的道数量
            int xh_min = txtHdr.first_xline;
            int xh_max = txtHdr.last_xline;
            int dh_min = txtHdr.first_inline;
            int dh_max = txtHdr.last_inline;
            int skip_trace_num = (xh - xh_min) * (dh_max - dh_min + 1) + (dh - dh_min);//需要跳过的道数量
            int skip_trace_length = skip_trace_num * dao_dataCount;//需要跳过的道长度
            dis.skipBytes(skip_trace_length);//跳过

            //读取一道数据返回 (分开读取道头和数据)
            buf = new byte[TRACE_HEADER_LENGTH + reelHdr.samples_per_trace * reelHdr.getMultiple()];
            dis.readFully(buf);


        } catch (IOException e) {
            System.out.println("读取文件头报错。");
            e.printStackTrace();
        } catch (SegyFileFormatException e) {
            System.out.println("读取卷头报错：数据编码格式及大小。");
            e.printStackTrace();
        } finally {
            IOUtils.close(fis);
            IOUtils.close(dis);
        }

        return buf;
    }

    /**
     * byte数组合并
     **/
    public static byte[] byteMerger(byte[] bt1, byte[] bt2){
        byte[] bt3 = new byte[bt1.length + bt2.length];
        System.arraycopy(bt1, 0, bt3, 0, bt1.length);
        System.arraycopy(bt2, 0, bt3, bt1.length, bt2.length);
        return bt3;
    }


    public static void outFile(String filePath, List<Point> points, String outPath) throws IOException, SegyFileFormatException {
        SegyTextFileHeader txtHdr = SegyReadUtil.getFileHeader3200(filePath);
        SegyReelHdr reelHdr = SegyReadUtil.getFileHeader400(filePath);

        System.out.println("读取头文件");
        FileInputStream fis = new FileInputStream(filePath);
        DataInputStream dis = new DataInputStream(fis);
        byte[] buf = new byte[3600];
        dis.readFully(buf);

        System.out.println("读取前端选点列表");
//        List<Point> points = new ArrayList<>();
//        Point p1 = new Point(); p1.setX(187); p1.setY(855);  points.add(p1);
//        Point p2 = new Point(); p2.setX(207); p2.setY(1228); points.add(p2);
//        Point p3 = new Point(); p3.setX(1026); p3.setY(955); points.add(p3);
        System.out.println(points);

        if (points.size()>=2) {
            for (int i = 0; i < points.size()-1; i++) {
                Point nw = points.get(i);
                Point nx = points.get(i+1);
                System.out.println("坐标点 : (" + nw.getX() + ", " + nw.getY() + ") 与 (" + nx.getX() + ", " + nx.getY() + ")之间的点：");
                List<Point> lp = CoordinateUtil.getPointBetweenTwo(nw.getX(), nw.getY(), nx.getX(), nx.getY());
                for (int j = 0; j < lp.size(); j++) {
                    Point pp = lp.get(j);
                    System.out.println("坐标点 " + i + ": (" + pp.getX() + ", " + pp.getY() + ")");
                    byte[] tttrace = SegyReadUtil.getOneTraceByteByNo(filePath, pp.getX(), pp.getY(), txtHdr, reelHdr);
                    buf = SegyReadUtil.byteMerger(buf, tttrace);
                }
            }
        }


        FileOutputStream fos = null;
        String pathName = outPath;//"D:\\SoftWork\\2023-06-05 物地震\\地震\\temp.Sgy";
        File file = new File(pathName);
        fos = new FileOutputStream(file);
        fos.write(buf);

        IOUtils.close(fos);
        IOUtils.close(fis);
        IOUtils.close(dis);
    }

    public static byte[] getTest(String filePath) throws IOException, SegyFileFormatException {
        SegyTextFileHeader txtHdr = SegyReadUtil.getFileHeader3200(filePath);
        SegyReelHdr reelHdr = SegyReadUtil.getFileHeader400(filePath);
        FileInputStream fis = new FileInputStream(filePath);
        DataInputStream dis = new DataInputStream(fis);
        byte[] buf = new byte[0];
        dis.skipBytes(3600);

        List<Point> points = new ArrayList<>();
        Point p1 = new Point(); p1.setX(187); p1.setY(855);  points.add(p1);
        Point p2 = new Point(); p2.setX(187); p2.setY(1228); points.add(p2);

        if (points.size()>=2) {
            for (int i = 0; i < points.size()-1; i++) {
                Point nw = points.get(i);
                Point nx = points.get(i+1);
                System.out.println("坐标点 : (" + nw.getX() + ", " + nw.getY() + ") 与 (" + nx.getX() + ", " + nx.getY() + ")之间的点：");
                List<Point> lp = CoordinateUtil.getPointBetweenTwo(nw.getX(), nw.getY(), nx.getX(), nx.getY());
                for (int j = 0; j < lp.size(); j++) {
                    Point pp = lp.get(j);
                    System.out.println("坐标点 " + i + ": (" + pp.getX() + ", " + pp.getY() + ")");
                    byte[] tttrace = SegyReadUtil.getOneTraceByteByNo(filePath, pp.getX(), pp.getY(), txtHdr, reelHdr);
                    buf = SegyReadUtil.byteMerger(buf, tttrace);
                }
            }
        }


        IOUtils.close(fis);
        IOUtils.close(dis);

        return buf;
    }
}
