import { SeismicViewWidget } from "@int/geotoolkit/seismic/widgets/SeismicViewWidget";
import { NormalizationType } from "@int/geotoolkit/seismic/pipeline/NormalizationType";
import { SeismicColors } from "@int/geotoolkit/seismic/util/SeismicColors";
import { SeismicPipeline } from "@int/geotoolkit/seismic/pipeline/SeismicPipeline";
import { MemoryReader } from "@int/geotoolkit/seismic/data/MemoryReader";
import { JSLoader } from "@int/geotoolkit/seismic/analysis/filters/JSLoader";
import { init } from "@int/geotoolkit/base";
// const sampleRate = 1;
// const sampleCount = 150;
// const traceCount = 150;

function createReader(sampleRate, sampleCount, traceCount) {
  return new MemoryReader({
    numberOfTraces: traceCount,
    numberOfSamples: sampleCount,
    sampleRate: sampleRate,
  }).setTraceProcessor({
    getTraceData: (reader, trace, traceId) => {
      for (let i = 0; i < sampleCount; i++) {
        const traceOffset =
          (i + Math.round(100 * Math.cos((Math.PI * traceId) / traceCount))) %
          traceCount;
        trace[traceOffset] = Math.sin((Math.PI * i * i) / sampleCount);
      }
    },

    getDataStatistics: () => ({
      average: 0,
      min: -1,
      max: 1,
      rms: Math.sqrt(2) / 2,
    }),
  });
}

function createPipeline(reader) {
  // Init JS
  init({
    imports: [JSLoader],
  });
  const pipeline = new SeismicPipeline(
    "MemorySeismic",
    reader,
    reader.getStatistics()
  );
  const colorProvider = SeismicColors.getDefault();
  pipeline.setOptions({
    colors: {
      colorMap: colorProvider.createNamedColorMap("WhiteBlack", 32),
    },
    normalization: {
      type: NormalizationType.Limits,
      limits: {
        min: -1,
        max: 1,
      },
    },
    plot: {
      type: {
        Wiggle: false,
        InterpolatedDensity: true,
        NegativeColorFill: false,
        PositiveColorFill: false,
      },
    },
  });

  return pipeline;
}

function createWidget(pipeline) {
  const widget = new SeismicViewWidget(pipeline, {
    colorbar: {
      axis: {
        tickgenerator: {
          edge: {
            tickvisible: false,
            labelvisible: false,
          },
        },
      },
    },
    layouttype: "inside",
    statusbar: {
      visible: false,
    },
  }).setScaleOptions({
    tracescale: 30,
    samplescale: 30,
    deviceunit: "in",
    sampleunit: "ms",
  });
  const css = [
    '*[class="geotoolkit.controls.tools.scroll.HorizontalScroll"] {',
    "   visible: false;",
    "}",
    '*[class="geotoolkit.controls.tools.scroll.VerticalScroll"] {',
    "   visible: false;",
    "}",
  ].join("\n");
  widget.setCss(css);
  return widget;
}

export {
  createWidget,
  createReader,
  createPipeline,
  sampleCount,
  sampleRate,
  traceCount,
};
