import {ImagePattern} from '@int/geotoolkit/attributes/ImagePattern';

export const LINE_PATTERNS = {
    None: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAIAAAAAQCAYAAADeWHeIAAAAUElEQVRoQ+3UAQ0AAAyDsM+/6QspcwAh2zXawGj6' +
        '4K8A8AgKoABwAzh+D1AAuAEcvwcoANwAjt8DFABuAMfvAQoAN4Dj9wAFgBvA8XsAPIAHMKcAEXKeuJoAAAAASUVORK5CYII=',
    Solid: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAaElEQVRoQ+3XsQkAMAwEMXv/oROSJa6RF3gQ13j' +
        'HEQgFNtw2TWAEKIJUQIApv3EBaiAVEGDKb/wFeDAQqAQEWMnb/QICFEIqIMCU37gnRAOpgABTfuMC1EAqIMCU37gANZAKCDDlN34BPpAEDVc2FJ4' +
        'AAAAASUVORK5CYII=',
    Dash: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAi0lEQVRoQ+3X0QaAUBBF0fr/jy691Y04HEasnq8Z' +
        'll1X++YhMCiwD+62msAmQBGMCghwlN9yAWpgVECAo/yWXwEeZYY1avOfwHxuHgJ8v30C+f4iVX0EKMD0AhTgIuaK//EV7yckff+drwoIsMppWCog' +
        'wFTM+aqAAKuchqUCAkzFnK8KCLDKaVgqcAJIIRwN6KSBLwAAAABJRU5ErkJggg==',
    Dot: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAeUlEQVRoQ+3XKw6AMBRFwXb/i4YgW/vEEQwWcZOT4' +
        'dO9XAqEBXa4bVqBBSAEaQEA0/zGAWQgLQBgmt/4B/C5Mtwo3T8D6XP2GPkA0AM4AjR9gQEIIIB+AY4Cv/rEO4Q4B6QFAEzzGweQgbQAgGl+4wAyk' +
        'BYAMM1v/AU90FARWj7aowAAAABJRU5ErkJggg==',
    DashDot: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAlklEQVRoQ+3XOw6AMAwE0eT+hwZRAoVlxdI2j' +
        '9bGi0ZDPnt5EAgS2MFs0QgsApIgSoCAUfzCCciBKAECRvELfwS8hjF8pf7Or+qnn1PNr+ryzwhUfF91AvoBowsQAQlIwLMV//d2awtwBMkewVxCh' +
        'u03rkeAgD1euocJEHAYqHE9AgTs8dI9TICAw0CN6xEgYI+X7mECN9g0KBED690OAAAAAElFTkSuQmCC',
    ShortDash: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAkUlEQVRoQ+3XMQqAMBBFQXP/QytiIUpgA/L' +
        'dZmyDWRhego7NQ6BRYDTONprAJkARtAoIsJXfcAFqoFVAgK38hp8B7osM71i9d8FxmQe05CLAG28JbGLtPQE+BATxIYi/D5gb0A1YfYFFD7SfkIr' +
        'felRAgFFem1cCAqyErEcFBBjltXklIMBKyHpUQIBRXptXAgd6wjANpuZVDgAAAABJRU5ErkJggg==',
    LongDash: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAkklEQVRoQ+3XOQ6AMBQDUbj/oUG0NEiWJVO8' +
        '9H/JeFLkPBwEhgTO4WyjETgISIIpAQJO8RtOQA5MCRBwit/wR8Dr5xjej8S+3cCmfAnYDfPpNg00uM50XwIGiX2UTAMNrjPdl4BBYgTsQfMJ6bHU' +
        'KSBAwACakh4BAvZY6hQQIGAATUmPAAF7LHUKCBAwgKakR+AGA1EQDdCHei0AAAAASUVORK5CYII=',
    DashDotDot: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAl0lEQVRoQ+3XMQ6AIBBEUbj/oTV2iglbjM' +
        'laPNvNOOTzJTiHB4FGArOxWzUCg4AkaCVAwFb8ygnIgVYCBGzFr/wS8PgYwyr1+v50ni437a/y1vcksN1/Ar4/wEqwak5AAm4dqARK5wQkIAFvBH' +
        '59xfITkp5X8hEBAkb4hFMCBEwJykcECBjhE04JEDAlKB8RIGCETzglcAKGBTAR0FL66QAAAABJRU5ErkJggg==',
    DashLongDash: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAAmElEQVRoQ+3XMQ7AIAhAUb3/odt0lC4M' +
        'UNLkORMw3w/qXhYCgwT2YG2lEVgEJMEoAQKO4lecgBwYJUDAUfyKPwJexRii1NX5i7f7egdX7xeP88QOHgRcBAwd3d0wBJwE3nDj/PpGMAFNwCjw' +
        '5xOwuoPkQyBNwC84jUpgBwECdlCVM02AgGlUAjsIELCDqpxpAgRMoxLYQeAG5aIUERB9l2QAAAAASUVORK5CYII='
};

export const FILL_PATTERNS = {
    None: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAIAAAAAQCAYAAADeWHeIAAAAUElEQVRoQ+3UAQ0AAAyDsM+/6QspcwAh2zXawGj6' +
        '4K8A8AgKoABwAzh+D1AAuAEcvwcoANwAjt8DFABuAMfvAQoAN4Dj9wAFgBvA8XsAPIAHMKcAEXKeuJoAAAAASUVORK5CYII=',
    Slashes: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAACxMBA' +
        'JqcGAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5T/wDltzBKAAAAQElEQVR4nI3PsQ0AIAwDwd88o2UURqCkQMIMYCOR61IkNiVHyzHkmHIsObYcC' +
        'vNYhqOE55MQcxAKNaF6ceT+c178tSs+7ibgDAAAAABJRU5ErkJggg==',
    Carets: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAACxMBAJ' +
        'qcGAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5TAP9bkSK1AAAALklEQVR4nGNQYMCEDAFYIEMHFsjAwogJGRgEsEAGDSyQwQULZGjCAmlgJC3cCQ' +
        'BFqCenjZUrKwAAAABJRU5ErkJggg==',
    SlantedBricks: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwA' +
        'ACxMBAJqcGAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5TAP9bkSK1AAAAKElEQVR4nGNgEMACGRSwQKyC/7EABg4sELsgVsuJFyTamdjdSQOHAgD' +
        'RijHZUjxFTAAAAABJRU5ErkJggg==',
    Bricks: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAACxMBAJ' +
        'qcGAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5TAP9bkSK1AAAAHElEQVR4nGNgEMACiRf8jwVQaCRNBLG6kwY2AQB8OzCZKCBODAAAAABJRU5Erk' +
        'Jggg==',
    Dots: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAACxMBAJqc' +
        'GAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5TAP9bkSK1AAAAMElEQVR4nHXBQQ2AMAAAsSMjC889kYEkpEw6BkpbcmLdSBsb+OPCB+3AmkgL60X5' +
        'AB52DQ3f6RevAAAAAElFTkSuQmCC',
    Squiggles: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAACxM' +
        'BAJqcGAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5TAP9bkSK1AAAAI0lEQVR4nGNgoAyEOmBChlUNmJBCeygFjKGYkIFpFSak0B4AHSAbi6G9BXs' +
        'AAAAASUVORK5CYII=',
    SquiggleDots: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQAQMAAACry+YbAAAAAXNSR0IB2cksfwAAAAlwSFlzAAALEwAA' +
        'CxMBAJqcGAAAAAZQTFRFAAAAAAAApWe5zwAAAAJ0Uk5TAP9bkSK1AAAAKUlEQVR4nGNgZMCElIJQB0zIsKoBE5JiKAsWSCFgDMWEDEyrMCHxRgIA' +
        '/+UbvSn9I4YAAAAASUVORK5CYII='
};

export const TEXT_PATTERNS = {
    'Arial': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAB0klEQVRoQ+2ZIU4FQQyGvyfBIjkAkgMA50BAg' +
        'iQEhUCSkCARGBBoQCBAg0IAB0ByABwIFJb8yTRpmn2bFW+ymU2fepvMdqadL/3b7oz8ZQRGjMBsxL1z64wACWBCMGoEEsBRw5+bRwCXgAtgH9gE3' +
        'geEyN55Be4GrNeSDeAE2AF+Br6TyyYYgQjgGnAK/BZfj4C/Cn4ngBWC2qLJCKCy0irwAJwD28BnBccSwApBbdGkB3ClSOgN8OT+e1nVmkvgBbgGP' +
        'oA94ADwEqxMeg+su6B4SU8AW6Slwpk9gILiymU9y4Zehg1SHcXqt1gDGnyHrobU2mNnOwGscJktmvQARuAEyVtoRnyWtMwYARRsW4AHN0KZALZIS' +
        '4UzG4BdWcvA+gLOyt4GoJ6tQ+7rggXjbYcMJ4AVLrNFkwZgBMX78uzkdgiAsf5T7fddakKT5QSwRVoqnFkAdmU62ypmxiEAdtWOKcEVLm8KJgVgl' +
        '/yabwannlXTLZfueJ4EP5ZBdhxKW4a1Tjgz4BToWYAPAlAZS0DM+yrhO1hJqZqPvhow2vOSvFveTwAXcHlTMCEANfNTQ2GNRvTLANJg2uaDfQD6z' +
        '3mypVmhaj+BafskgFOgZwE+/AO6bokRdqnAggAAAABJRU5ErkJggg==',
    'Arial Black': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACvklEQVRoQ+2ZrW9UQRTFTyW1VBNSgyQ' +
        'hdaChuqaCejQJFkT7B+AICBIEFVXV9cURElyxOBKqa5uTzE1Obu7Mm5lud99uZk3Tt/Nm7sdv7tduYXyGBVZoga0Vnj2OHhbAAHBAsFILDABXav5' +
        'xeATgcwCXyTRfALwFcNNgqtcAvi/pfT0rEvE3gEMAf9KXTwCcAXgKwH/XoOLk0ocATgG8uudzJgWZ+4IIwPcAjpPgPU6aE4BUQ3UYAM6MSA+gOsh' +
        'E/QDgZIlytwCsa3OgUfSjFJEGgEt0ZM1RHsAopV0A4PNr2VDX0bm7KWpy7TmAz5kUrNFV5XsB4Ed6sAgANQXWAlgjm8msZYo9UzvlUrA/Q/Wu8df' +
        'GrVEAvdH+AXiZNPaGUkhYU7HOMkfzb1QD5hxsRrUzFgGgj+S2dy4C1spGWUtrLQr/D2rAPbFLdKk3Dq4ahRRA33z8AsAmhB/fjORSH9dGAG1ninJ' +
        'da6m+F8Ccvnp5IgAjWNi0RLLlAFbbUY9PTl/+b7bsqatrfLmWawzABwA+AngjkexnoWMsQVIDkE+RCvlBQxc91QVz39YmpFa2Um0c7WGALLumnjW' +
        'YBmB0s/86KNVwUXQwRXMATsFiUbYXQAXNXyjb+1HmUtXK9k4mBL0AjggoV8IAnHIAX9G6pRXAZzJb5F6ldLsIAH0pYLLvBADymc09W2RrBVBr5Z7' +
        '56qwjWa9wBLCULvy+Ns5oBTCCintr2r/PCFgCUJsDkyEnm14kjWS+fubY6msaRHMv1qFXUhfaM+v8e/239u8RQDVelB6064sg8ZEgSsGa+nJGuyu' +
        'AJWeU5oB8z34dmZKNvwjVdMwKm9q0pj5ee6haFCCAnNlZ8xGlBj/z4m1+LI1CDYB0nN+HZ/Hsb+mnMYtS+wtuQgw+2qWmi7WGKJLNZqG9c0A/HlL' +
        'ZWvy2MWtvAUCtUyAuXJypAAAAAElFTkSuQmCC',
    'Bookman': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACV0lEQVRoQ+2Zv09UQRDHP9fSamdCYaOdDbE' +
        'RG2OC/4GFUlMQSCwoIQZKChIJCZZGLSyk1h5aEjrt+AOwtiXfZCaZbO7u3Y8Ny73MNsftvLc7b97n5juzDMiREWgYgUHDvXPrjAAJYELQNAIJYNP' +
        'w5+YJYDLQNAIC8BTYGOHFe+B7RQ8f2HpfK69b0cVc6i4j4BnwBbALvAP+mQOaOwf2gIMKTjl8a0BtsCu4l0u0iMA4AOWPoBSIEcx5/MwMOE/0enh' +
        'vF4ACbwd4C/yt8PwJYIUg9mmJcQA+AX4APwsJjlKqWHwGPgD/LTDj7CWAAvyb3SdZ/hVqRE3LdmU/gIdWEmh+Fbiw++IamoryLts6sA18AiT/tUq' +
        'KPnHQ7FkigKr3ylHWag6QakIB4N+vDcIlA6jLribkDDgC9HdcS5A4SH6NmiQHR1C9tP2WDSwBpgztwOnzTYD7t9meAicVM3qzF9eXjSfJgHpWl+D' +
        '48j3jqUb0l7oS4BhmvzFABcRz4GMh7cMkWnWohjdCETJvmLxe3QccNtnKa5XVlQkd2L68x4V9jq4asJRhwfCokFy/ZhN41WH/YwDGLBePeWYB0CV' +
        'YsuwgetOUAN5zNKcFcFhTErOKMmDZtES7Z0DJrmRbsh/ruWkBlKTG46PyOCkBXGAAYzPhkJQ1n2Q2SuSkdj+ILgGZBcBY05XHRgngAgA47j8h3oH' +
        'GI5iyyy27ylH2cl4NjoZ3wV9Mvl+HJuQxoLpOQ932pX3qu2q9LeDYulvNyRddL7+17qHdq2tVQwrWZzYXM+89f039de8WI+LHET+7z3IAAAAASUV' +
        'ORK5CYII=',
    'Calibri': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAABw0lEQVRoQ+2YL0gFQRCHvxe1GgWr0SqoxSI' +
        '2kwY1iyiCWZQHYjIIoohdg80mZrXarYLRbJUf7MKw3Ht3nPdY795sO27/zezH/Gamhw/3QEYP9DKe7Ue7B3AAHYKsHnAAs7rfD3cAnYGsHmgKwE1' +
        'gCTgEfhKLFoBjQHO+s1rrh/87D1gAp4B7YMXcchF4q3BrC+AM8ADshbV1AdSe2w5uBe+3eEoEcDZAcx4glEkC8gw4qhC5HMAWQ5Dz6hFASaTGac3' +
        'LuATXdNy4LxOAUXoF3zC5FWR3A+TZAjgZomjcL0rwDfAY1j8baZ0ALoAnYDekAJJ+Dc8dO06oAJT8XgIHwMcAewXRsomQaX5WBuArcGuKFIE1Hb5' +
        '1pACcBzbMHermjh1/sm6ZVxXA1GpB2wf2Q35YBmAaySz0nwHAF5N/6jwHsFusFVoTAbRVa9HEKJM75qeVUQdwDGAZhYkCMML1NaQIsZKpPt9fI6C' +
        'i23WQXI+Ao3jZluwZq2ABoTxtK2nDXAWpXU+qZAGpNbG5XBYBtfdJADwFPn67BLcEmiavaRvRsRc4V1Dppk3qNWDV9AjLAFRD+T0UItreFiQOYJM' +
        'v2rK9fgHtF40RvjfO9wAAAABJRU5ErkJggg==',
    'Candara': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACDUlEQVRoQ+2YvUrFQBCFv1vaCVoIPoB2PoB' +
        'a+wgWYi2iCD6AP2hpIQgK1mphY6+t2AqWNnaWVoK2cmAHliWb3BtIluCky81m5+Tsd2d2doRf7kBBB0YFY3todwAH0CEo6oADWNR+D+4AOgNFHSg' +
        'N4DJwAGwAX0Wd8OBFHEgBXADugaWg5hrYB347UucAdmTsUKaNAVQmmu8YuNQXB3AopHSk0wBUCVztGT59kgPY0cIOZVoBOAPcAafAS0a4jVmLnsf' +
        'lWaX7AjgJezobl5bwtMSr3E+HPeAPcA5sRTEeo/2hvXsEbAOKcQhcBf05bUNZi3+pUwBqYY+B3QkaAQPyJiy+3ctEaygMmLMwxu53ItA1djPThEw' +
        'FID/Dn6MqRtWipdr+5cIO5aMNQGWvPeB9AuHaM+pS5sxlUY35CADG4y1ME/zx1kDvKEM+h/nqpFbFmuDTfGhfDoxbgqUnLZ/6TSVwHAAfAjzKmHG' +
        'ZTwGsK/V1ANZp68tLj9PCAWtClDHUEOTO4/TsEliPsmRTBrQSqoxlAKbZKwZwNhwBpSXamqMcgE3aWtjir/TlgAFomWcugcx0pF1ybn8WNzIxgGp' +
        'yqkCJu+DFikPp+GgoB2CTtr689DgtHEgPorWYt8k8K8Br0qG+AU/Ad00JTgG0BiWeX3Posv2ngFMnbZdKvJ1N5gC0ONY9p9pa2OKv9OXAHwgglRF' +
        'M2ETEAAAAAElFTkSuQmCC',
    'Comic Sans MS': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAAGWElEQVRoQ+2ZBaisVRSFv6diYrdio' +
        'oiBKBY2tiImdnd3t9iFPjuwuxVFsRULFFsRuwuxFTv5YB04/M7ceG/evXNhDjzmvpnzn9h77bXX3v8oeqNngWG0wKhh3Lu3dc8C9ADYA8GwWqAHw' +
        'GE1f2/zVgBcBDgCWBP4FXgd2A74eByYa3zgSGB64JDsNzbbjAesCOwDrAJ8CzwMnAO8NjYLd+jZ+YELgYOB56s1ZwOuAC4Hbmqz15y514bANMDTw' +
        'KXA3cBfHTrfkC9TA9C/Bd1FwByNkyyXC3f6gJ0E4GQJHIOnOU4Hjgb+6PQFBrne/sBZwAHA6OrZ1YAHA84Dgd+q3/TLBnmu6ZfHgW3GETkM8mpjN' +
        'r0G4OLAbcCngEZ4DlgoRtkLeHnMthiSp7yHrKeDjwfuBL4bkp0HvsmkYeKdWgCtAPMBYEvgm2rZ4pc7gPOBD4F/Br5td88sADQFXgLMAmwPvNHdx' +
        '/7f6SYJo5imdOY7XZiWtO31wIQJjq3z6f9PBTz7DJE771Y3FJBnAHsC93dApnSVawsAdwROALYCHu3jhBMBapB9gQXCmMcCnwCmQCNUIz0DmPbmA' +
        'ox499kBkAXOrfTYFMDFccy91b4GxEnAZoBpxrTlZ7vIN5V7DtNsPR6JpLgH+D0/NHWijKIuuzrO3RZYBjgc2BQ4NOf1zu9Xi08ALAysGrvd10+ad' +
        '+41wGWxoYH+UfTvlcCNwO6x7QvVPmpZGX3y6ru3gKuiGb/qKkQN8jACY+pcRONaEBRHNZfS4KZmo7UeGsIUPXGA9EQKgdUz6RVgqkpX3gzsAvwIT' +
        'Jtnrs2njwjK86Jtyj4/RQcJqHbDAFAPCaClGpMOA87MdzKkQLVAkSk9w0zAOoCOdw3B8VLYtCyltjwlwbR+/p6v2kd2E0CetdVYN+sdlWC3EHG/x' +
        'VKUeCbvbTBYWJTRDJgaiHcBuwJfDtLvXTNdABqZaj8p/qE+TrYgcCvwVICqTtkiwl+m+iwgWgN4NsbW6ILtxFR5m8RgzjfNtAKgbCx77p2K0Dk+f' +
        'wvQFwCbRzctL5GzWmVuDPiddzVoTGs/A55R3SjbySymvOvCTlbmMvPJgOwveE2TN+S+gsUA+x74tw/bydBmGJldALq36xqMAsjvLUoMEs9QFyiti' +
        'MBiRH95HjOM1fOIHAJwWeCCygHtLqKDSlrSSI6iazTqizGcbZyiI43qWWMoWzolDQkyo18G0Qk6RAaZMinKtQTh3x2w6qLZwzMtCcjM6q+vs7aAs' +
        '81kMJnOvKOA2y9yQWDV95g96yk7rFxth7zaj+Y0y8jyzjWDKBdsnRTQ3Z7gdx/ZuFkJtzKDttJv+kJwj8hRAKh+WavRapEtNISpTYPJILWjmgDUC' +
        'RrEtFYMYoSqYYrgniesJuOYZpoALL+rt3TWQIYi3v0+rwBTP7dy2FfWtd0xb9ijpMoVArACQEHgPP9vR8BRA/DPsObOCV5TokBUhnj3VqOZZQzmt' +
        'cN42lY7qQebweAcbWdqb1b1AlU9qZ2UBiNyCMCZI47VfhrivVRkpr3Nw0Q6YPmkHrWQKcqUZD/L3qHz1HSymILZitqhoW2P6EwNXFJumdME4HRhC' +
        'tlFx3yRlOf6suKTLays9jR16Sz3sgiy3yfr+N1BwNtJdUoAGc60ZbHlHdR8pjN/ezNgk+UMgl8qAAoin/uhOoPgnxs4LYVXOyCYZus9XEuGfyyFm' +
        'nY3Q5iNlDlFj2o/G9PHpYUjCN1TJveufm7UaGqPKCAKQP/pYKvRWuB6EUWxhYlsoUMFlkxYD3WSVaqFhgBUH1ntOZrpvfTC1IsaVcA5X03ms55FR' +
        '5kC67O4v1qy3VsC335YxTYbtZ5BhjDlm6pKT63VvKKlmrLBNQSpILLAkfm8c3M0m8vl96L/lg7DydQl0Aze3aqALRnANxzaesZ8rtdiP4sobaWE6' +
        'Ut/djUgSxvGSstUdUyYTrEvq1iU1K95pH0ZRT2lXlLDCB7Z05RtVOto2zEO9eDZYQ6LjtIukXl0tOzVbMNYbVsY7JH0bQulvzaMe9WvqgSYbRudI' +
        '7BlZ4f3XSmFk9LAtT2fjrZ4Mo0KMgsu72mR4nCugagEUXvJjrZf3MdqWfCbDsv82ukylmxuISRgDKbS+xOABrSvOx3FhmpfK3ffiDSre+1rAaPc+' +
        'WAkg88L/wcEjo0g/sDjmgAAAABJRU5ErkJggg==',
    'Courier': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACL0lEQVRoQ+2YO0seQRSGH0trWwux0c4foCl' +
        'Mo/gHLDSlBEwQAraKl1YQRERrTWFjKaSxMfZa2kgKW2tbeWEODIfZ/dzPkSVwtpvZmTMz7zx7LjtCPKFAjwqM9Lh2LB0KEAAGBL0qEAD2Kn8sHgA' +
        'GA70q8D8BOAt8A34Br72qFotXUyAArCZlGBpGgRzAMeA3sJAMnQGnwDxwOIzxmBMKDFLAAJwCjoAN4DFNUsj7C2wD+4MMxftQYBgFDMA14Ap4cUY' +
        'E4VcH4Bawl42bA+5S27yomivJnuC+BGaAfKzeXwB/0tjF1NbcfJx9COqXV27KAf2+/Iczmjz59/RRHWRt24M//zCaxpwOCghAQbMO6EIGJfe65Kc' +
        'UqrWMXeqt69sEThzQmnuTwWrzdxOc8rbytLIpaO/dOdr2KZj1KIWwR31fCsDKttacBn6k/YynSc8dtIuhFRToAqAubqmQDwqYHDjftm02Aaj88ty' +
        'BWTpaE4BN65kX/uds6xw7wM+Cx68gaZjookAXABUKJ5yXsbUUFq9T/tgVwJK37AKgL5783FW356YPqYtuMbaSApYD5gA1mf4sD/hRANs8YOksAWA' +
        'leGqYMQDlRY5TaLIq2ELYZFaElEAt5V95uM2T/7y4sBzwowDKTqlYUr/PWdUXANYgp5KNtv+AWsJXkqVwV/pNk1e+siPwVE2relZIVMWt3E8VaVv' +
        'IbAuvviK2qjq3l4dfvycb9wAsZ7+fKkkbZt6jwBs3TI0RJcyP9wAAAABJRU5ErkJggg==',
    'Georgia': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACfklEQVRoQ+2aMWsWQRCGn5TW2tnYpbSx084' +
        'itQZioQHBQkEQBC1jSFKmEATFdIJaKPgDtFXB1tIqglgI2sY2vLAvDMPdeUnht5fbrT7u9nZnd557Z2bvW6K1tgML3IGlBc593Kk3gFXgGvDtuIO' +
        '05+rYgQzgMvAGOJ/Mew9cB/4s2OxTwGPgXCX2LHg7pj99BFDKsg1cAj6XpTWHT9/HVa/AAErdXiX4bLggfAA8q0ABq97MZtzRd0AAngZeA9+B+8D' +
        'fEcNcBD6FflE1dTnffwTslP6G/SuwCWyVkO8xrMTZjMvAGnAbiCmB7V8pD2jclh+OcGINXQSgYYmQDNmm/k+Dk/28ARJgD8N9A6KwniHcA54DL4D' +
        'dMul6yO8E49n0Ymj83Ec26PpByRE11NiXqQY/zNaGCOCNooRDm+Gc8GPo62s/S5iWmkbYNF6GMkPsOTNwGTaPFQGM9tqWBuBEkB5SwBxGtaQ7wBX' +
        'A4S4uUwr6tlTRd0MhE0OyVbIPwAzqGAWM4yv8/gL2mwJOg8AxOWCE4ndRvpc9auljHIVTKaFbDtN9AFrhVBCpdR3/DIVgHRN1QTsNb8zQSlfBXUc' +
        'w3o7o8H/lWF353lFCsMBUeB3K36I9GlugxxeiATghkA2gcydVmDkXzIpjNYv9pHy3SlV7NR3pdEE5pIBdVXCcq0uRnXNagb+0EDwNCvOXkK68T3n' +
        'VTeBHOAfM/VTNRtXK9zNADrHapVh9Cz41V8v6LaieAPeACwVuXfdxy5lwJCQ73gEfesL3NLwyIytr+hZs9coFjBVUUPoLzYxcdLKXWhOAQ/lj37H' +
        'LyfbODFZXE4AOt/nPELX8EWIGOPz/JR4CB0XAEaDbLb8AAAAASUVORK5CYII=',
    'Impact': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAABw0lEQVRoQ+2av0oEMRCHvyutfQHxBWyu08rG' +
        'R7DQUrBSsBX/oIKVICgoV1gIWijYq4WVtoKlD2FtKz9IYAi523Nx9zYyaQ4uk83sbz4mmWF7+HAFJqhAb4J7+9auAAJwHngNWjwBK8CXa+MKtKFA' +
        'KQBOA7fAUhBlAXhrQyDfo1kFHMBm9fWnVyiQA3ADOA/Z5gPYBw6AOWAVeDTZaA84Csf2TdhrALwD+tX6ZeATmAJOgXXjU1yvv3Lz2k+Zbxc4NOvs' +
        'cz3IBStQBWDVq0UQ+kAEMF0jyC6SI9TaaP4kA2e0EYSzDmBVKMqcrwJQRckmcJbcvxYNELqPzRgAlfkegGeTES+B65BFBdyLKXxkb+fTPTWvLHvl' +
        'd8AyIRvltQP4/2Ja1Bu1BeAW8D3knpfLgGkryKvgorAa39m2ANwGjk0BcheKE3maA3AN2An2fgSPH8/iLNsC8Ld3QAewOJTqOdwVAG2RoSLEApir' +
        'kr0RXS/enVvVBIC5Nsw9oGNXvcR06IhNj2hrI9jUV7Q9RO8Ddg6leg41AeCwRnRsJqfNbdmrSNFIG9U206kwib1GB7BevDu36q++hrFwRKBU9fpw' +
        'BUYq8ANfnMgRwWEk7AAAAABJRU5ErkJggg==',
    'Roboto': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACDklEQVRoQ+2ZsStHURTHP7+RlUEp2diUbFgs' +
        '/gQDZkkpg5GB0aCUYpbBooxm/AEGxSalDMxGOnW/dbvdS++Xet575y48z7n3nnM/v3PO9/56+PAI1BiBXo1r+9IeARxAh6DWCDiAtYbfF3cAnYFa' +
        'I2AAngBrmV3cA0vAU4UdDgHnwFn4WcHU/7WLEVAGnAV2gGXgAxBIIxUglM0isOIAdhGn6j6XALSZDMrbijD9RQacACaBq+ruuEXTIvDfABwADoEb' +
        'z6BNQ6m//ZYAtCx0AbxFZVkrxKXW/nYKbAGfUem+BKZDb5n2kiX7sbDmVFhIdi8BSvWp15k99ee9W9UegRhAK7caJQEiePaBuwi45wDhYMhc6h0F' +
        'z3iAxuY3kfKbvURMmhH1rPmsX/XR4AjkMuBwyEQHmTJoImU+ynjqFY+DWHnPqGBl0w3AwKlin4ojWy+ezz4EPhocgVIJNtC2MwrYlPJoAmAMxGMG' +
        'wDhrLlS0t32sJiX3L4ROg4+sXVsvAahSZ96qv7Pfc2AagEfAJlDKgHo/kwH7J3sp8blQ8m0PaRvQrhPpmDcxgCqjunjWQQu8+H5QPZ8JD8uKNqyv' +
        'k42VRnsWyK/J+5J93PM9AF/AeijduqPMtQEdO7b2uJv7JiS+RFYGMo+ldiU07MLZxm6AS1FJVW56KZ2+T+0N6L1oXkEpFRyr7vacREc9+QZO2aYR' +
        '7XZ7ewAAAABJRU5ErkJggg==',
    'Times New Roman': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAADwklEQVRoQ+2ZT6hNYRTF1xsypQx' +
        'MZMKMZIaJwtC/YoAxIqUYkj9TpRRlYIQBRRmiTFAGEgPFTEoZMGWqX+1Vu9137rv33Xf/eJ0zue+c+75v77O/tddee98F9VcfgRlGYGGGtnvTfQT' +
        'UA7AHwUwj0ANwpuHvjfcA7DEw0wgAwB2S3izixU5JHyTdlPRa0sMZeX1M0oOw/VwS97/jflX4dzLu70o6L+nvBHxdEzHY19i7+jUB8ytnSwDIIXI' +
        'BKh9iBtklSa8kvZ2j1zYQL0u6XvwioTZMKUnwY1cC+iZJjyT9LMkxR6GbL1cA4J5gN5ikBUCCyvOPc+T6fknrJMFyxwvYACDXNBKmAhC7Tg6qxjR' +
        '8mKNjGd2VqgFbABx918mvAIBfJB2RdFjSUUlfw2wPwMnHf9ksjAJA6x5KHpnt+/vhDdrsU4BhbdKVmQmqTsvslcsXNtjjWcebGoDfQ/dRcq0HKwB' +
        'bNp8mvWifAfO1sGe/kB88a5V6s10uwdbTLf2Z9Strc1z47oSkc5JuSUJbYvNG8rPuad8cIu836FycpMsGoHE3GhaAWXTzorAPmtEinAPLh+oDyyU' +
        'KX69KuhdslQFNg+PvfsXeALur2TEACaiB+yT0YAYg4OuySRJxiFzWkazlmcGMj9skvewIdAVVVwNSS3Uu0yRPbaw2RwI7ORwTJ3/1E5/Xx3sQ39a' +
        '5/Gjo5XHxM/b6YQGIoUEMaKDUA3Vm8+mgVqdzpg8bpAxA9jPzkAjfkgbs6vCdIPUg/f9mk63RRXcxRwYWQGWaULVfjRvumZX9vjlOaHGSCiaEEbG' +
        'dWS0nZWZ3M+TqRgLXcxkbOMu1wbQBSJnpGo1ksCw2yqgAJB4EGT14R9LnkAnsOchmZWH+lwsw4edeSS8GjHIqs+EDNvN4yAx9pjQlZi3sHAo/vW4' +
        'YAPp90b/bUzfeAzBRfWXAXN66kmiYUUYLgJkNzEKV4Vo2zQ6Pg5mYG8I+aD8Os6v8tjSgAU15d1lvPXPCuPyPyoCtkm4tuiIA2CqFtQy0ysKgEvw' +
        'nBDVBNwtaY1G6LgR7uQRdkXQ2DZozeFoA5PvKNgZly6aBZZCiYd1QMXDn4nOQcK/AyT68S+9ZRzOV4ZYCQNiadTWu/y0Aq17KZbBO/k9JOljE7sb' +
        'URaJHaCz45PJe/J2bFz8niACQErol1rTmaNWPrkE0W3gGV9fU8t5KpNZ8LydA65eQ2tW6scgNmZ+5uQDc9dcdmPN2isMBSadLrN/HwJtYsRcy4WL' +
        '8YICfu8NZNHE9l0n9OrQkWfgPnwRrIOzXC8gAAAAASUVORK5CYII=',
    'Verdana': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACXUlEQVRoQ+2ZO0oEQRBA34bGhh7A0AOoqR7' +
        'AwECNBQVB0EhUVIwUBEHRTFADj2CuBzA0ERPBxNhUCragKKp7B5ZxmqE32aWnP9VVb+s3A+qnaqBDDQw6PLseXTVABbBC0KkGSgFwBVgD5PunU43' +
        'Uw/9VAwLgPjCbMf4k8Ai8AictSVcBbEmxpW8rAAp8L8DcEDIvszy/ApaB95YuVAFsSbGlbysAjvJw4iGngG3gt6ULVQBbUmzp22oOmAJA4bwfhmG' +
        '9j0B5bC63GjwXaG+AO2AGODAh3K+XrZ5NGjABXADrToH+HJV7C7gEFofz7VkypPfQ57ptyuuXbrfeyKcATgNPwKYLwz78KhiiAPWIuvbMQKiA3Tr' +
        'PGa2XvfwfQObtANemKInOkXUPDt5ongC4AZwbLz4q9eiNkUu+iAIYgRGNpTyljM8b2FJhO5VPNgnBKs+X8aTRumheZIOUdy/ZXr2TzbZhxJi7ptj' +
        'wniRnWAFLoNM2SgrA1HgKwChUW6+aA9B6afmdCus+XPfOyCVfyALogfNApgyo93sz8I4LoMryHeSFFqymAGq4tfCOKr5KtltvZLMA2pC7B5wCNtw' +
        '1DW2inHEBjHqTTVMCP0/kkYLGe8QKYAEY+zch6vUOgaOgKGmSq+UAzOWQ9k1IBPC4ANo/k62M22ywF2DiskXwAGrok7aJbYvoLdRrfLrq1leZKQ8' +
        'YeR0Nj/a8qFiJKuumITiqsrXNU3PADhmN3gWroXOGiYoD26PLNa99T07O+QjeBWuLRdWzACy5UNoUQPXK2ruUfFVaTiJn9YAdAvgHeovoEWdFa7w' +
        'AAAAASUVORK5CYII=',
    'sans-serif': 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAKAAAAAQCAYAAACRBXRYAAACcklEQVRoQ+2ZvS8FURDFj5KWTqehk2hR' +
        '0ysUiFJEIvEHUFCSSMRH1CgUEq1KgU4kSn+AUq2Vk9xJJifz3t7GrnjzqvfcvTN3Z3975sMQ8pMR6DACQx36TtcZASSACUGnEUgAOw1/OvcATgK4' +
        'BTDtwnIJYAfA9z8O1TCAYwBPAG7Kfc4CeC7fV93f/3EYurk1A9ACPgfgxR1lFwDXVgB8dXPE1r2OFuCuErzfj70BSND4ORCXg/gwBvGef5+0Hh4I' +
        'oKWgzwDAaJtdv+EW99xeKiaBvgBw766J1HXfrdekOp8auTUqEejb7L4DWAbwUfwQrlMAj2Uv19cBbJYU/CplyMOAqX/rIJoCMsVeA2gKeASr1Y6H' +
        'JWUZJB4OTeX8Pe7qS7OxJSWADwjtnjmgorOoHy0tTN1o18oKrQFTAVvE0DchBqF3r6pEUE4AbDtVURAUFNrzgL2Vor9Wce08PN9aH0XqBbGHfSSo' +
        '7xLAFoFTV9EYRlOspjGzoV2zpWFLwb5xUThqFdef1/vTdG6KFgHqz8Pr2Omy1rVmKwH8YwBGD/2uPDSFk9AdiaLVAEgfNfWchsbS40JZ8C9HpOC2' +
        '30qLBLBD2CLXNYNon8JmpA6jzSgFc08/BdSzGIwE+ryolEHGayPF03pusSFF047tSQX8IyASwKYGwI9oCNW8DKdtv6lkrQJqCCLbTWHyNelY8HJY' +
        'arYzWw2YADZFtqV1PwdckpGFNQ++6dCu0qdk63qpkv0UMGpCajrPqAnRblq7YH25UgFbAqvWjU/BWpPRRjRn01qLnfKE+4/JVAOALP6bZom9zq++' +
        'o7GRnwNq+k4Aa8lo6bof11nuESEolD4AAAAASUVORK5CYII='
};

export const FS_KEYS = Object.keys(FILL_PATTERNS);
export const LS_KEYS = Object.keys(LINE_PATTERNS);
export const TS_KEYS = Object.keys(TEXT_PATTERNS);

export const IMAGE_PATTERNS = {
    None: null,
    Slashes: new ImagePattern({'url': FILL_PATTERNS.Slashes}),
    Carets: new ImagePattern({'url': FILL_PATTERNS.Carets}),
    SlantedBricks: new ImagePattern({'url': FILL_PATTERNS.SlantedBricks}),
    Bricks: new ImagePattern({'url': FILL_PATTERNS.Bricks}),
    Dots: new ImagePattern({'url': FILL_PATTERNS.Dots}),
    Squiggles: new ImagePattern({'url': FILL_PATTERNS.Squiggles}),
    SquiggleDots: new ImagePattern({'url': FILL_PATTERNS.SquiggleDots})
};

export const LINESTYLE_PATTERNS = {
    None: [0, 10],
    Solid: [],
    Dash: [20, 4],
    Dot: [2, 4],
    DashDot: [20, 4, 2, 4],
    ShortDash: [10, 4],
    LongDash: [40, 4],
    DashDotDot: [20, 4, 2, 4, 2, 4],
    DashLongDash: [20, 4, 40, 4]
};
